import React from 'react';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';
import isEmpty from 'lodash/isEmpty';

import {
  ALERT_TYPES,
  NO_POINTS_POLICY_OPTIONS,
  NO_POINTS_POLICIES,
  RESOLVED_EMPTY_POLICY_OPTIONS,
  RESOLVED_EMPTY_POLICIES,
  COMPARISON_OPTIONS,
  convertAnnotationsMapToList,
  getAlertType,
  getAlertTypeTitle,
  getTitleByValue,
  THRESHOLD_TYPE_OPTIONS,
} from '../constants';
import LabeledValue from '../../../components/LabeledValue';
import AlertProgram from './AlertProgram';
import Panel from '../../../components/Panel';
import LabelsList from '../../../components/LabelsList';
import { formatDuration, formatDurationSecs } from '../../../utils/duration';
import { ReadonlyTable as AnnotationsTable } from '../components/AnnotationsTable';
import EntityInfo from '../../../components/EntityInfo';
import { makeOldUiGraphUrl } from '../../../utils/SelectorUtils';
import Markdown from '../../../components/Markdown/Markdown';
import ShowMore from '../../../components/ReadMore/ShowMore';
import { ReadonlyTable as PredicateRulesView } from '../components/PredicateRulesTable';
import AlertEvalStatuses from '../components/AlertEvalStatuses';

export const ChannelsBlock = ({ alert }) => {
  if (alert.channels && alert.channels.length > 0) {
    const channelsContent = (
      <div className="table-responsive">
        <table className="table table-condensed">
          <thead>
            <tr>
              <th>Channel</th>
              <th>Notify about statuses</th>
              <th>Re-notification</th>
            </tr>
          </thead>
          <tbody>
            {alert.channels.map((channel, index) => {
              const { id, config } = channel;

              let statuses = '(not selected)';

              if (!isEmpty(config.notifyAboutStatuses)) {
                statuses = <AlertEvalStatuses values={config.notifyAboutStatuses} />;
              }

              let notifyDelay = 'Never';
              if (config.repeatDelaySecs) {
                notifyDelay = formatDuration(config.repeatDelaySecs * 1000);
              }

              return (
                // eslint-disable-next-line react/no-array-index-key
                <tr key={index}>
                  <td>
                    <Link key={id} to={`/admin/projects/${alert.projectId}/channels/${id}`}>
                      {id}
                    </Link>
                  </td>
                  <td>
                    {statuses}
                  </td>
                  <td>
                    {notifyDelay}
                  </td>
                </tr>
              );
            })}
          </tbody>
        </table>
      </div>
    );

    return <LabeledValue label="Channels" value={channelsContent} />;
  }

  if (alert.notificationChannels.length > 0) {
    const channelsContent = (
      <ul className="list-unstyled">
        {alert.notificationChannels.map((channelId) => (
          <li key={channelId}>
            <Link key={channelId} to={`/admin/projects/${alert.projectId}/channels/${channelId}`}>
              {channelId}
            </Link>
          </li>
        ))}
      </ul>
    );

    return <LabeledValue label="Channels" value={channelsContent} />;
  }

  return null;
};

ChannelsBlock.propTypes = {
  alert: PropTypes.object.isRequired,
};

export const DescriptionBlock = ({ alert }) => {
  let description = null;

  if (alert.description) {
    const descriptionEl = (
      <ShowMore>
        <Markdown text={alert.description} />
      </ShowMore>
    );
    description = <LabeledValue label="Description" value={descriptionEl} />;
  }
  return description;
};

DescriptionBlock.propTypes = {
  alert: PropTypes.object.isRequired,
};

export const AnnotationsBlock = ({ annotations, label }) => {
  const annotationsList = convertAnnotationsMapToList(annotations);

  let annotationsBlock = null;

  if (annotationsList && annotationsList.length > 0) {
    const annotationsTable = (
      <AnnotationsTable
        annotations={annotationsList}
        limit={5}
      />
    );

    annotationsBlock = (
      <LabeledValue
        label={label}
        value={annotationsTable}
      />
    );
  }
  return annotationsBlock;
};

AnnotationsBlock.propTypes = {
  annotations: PropTypes.object.isRequired,
  label: PropTypes.string.isRequired,
};

export const DurationDelayBlocks = ({ alert }) => {
  const windowSecs = alert.windowSecs || Math.trunc(alert.periodMillis / 1000);
  const delaySecs = alert.delaySecs || alert.delaySeconds;

  return (
    <>
      <LabeledValue label="Evaluation window" value={formatDurationSecs(windowSecs)} />
      {delaySecs
        ? <LabeledValue label="Delay" value={formatDurationSecs(alert.delaySeconds)} /> : null}
    </>
  );
};

DurationDelayBlocks.propTypes = {
  alert: PropTypes.object.isRequired,
};

const MainBlock = ({ alert, alertType, isMultiAlert }) => {
  const isTemplated = alert.type.fromTemplate;
  let mainBlock = null;
  switch (alertType) {
    case ALERT_TYPES.THRESHOLD: {
      const alertContent = isTemplated ? alert.templateData.type.threshold : alert.type.threshold;
      const aggregation = getTitleByValue(alertContent.timeAggregation, THRESHOLD_TYPE_OPTIONS);
      const comparison = getTitleByValue(alertContent.predicate, COMPARISON_OPTIONS);
      const predicateRulesList = alertContent.predicateRules || [];

      let predicateContents = null;
      if (predicateRulesList.length > 0) {
        const predicateRulesContent = (
          <PredicateRulesView
            predicateRules={predicateRulesList}
            limit={5}
          />
        );

        predicateContents = <LabeledValue label="Predicate rules" value={predicateRulesContent} />;
      } else {
        let predicateExpr;

        if (!aggregation || !comparison) {
          predicateExpr = '';
        } else {
          predicateExpr = `${aggregation} of points ${comparison} ${alertContent.threshold}`;
        }
        predicateContents = <LabeledValue label="Predicate" value={predicateExpr} />;
      }

      const { selectors } = alertContent;
      let selectorsEl = selectors;

      const oldUiGraphUrl = makeOldUiGraphUrl(alert.projectId, selectors);
      if (oldUiGraphUrl) {
        selectorsEl = (<a href={oldUiGraphUrl}>{selectors}</a>);
      } else if (isMultiAlert) {
        selectorsEl = (
          <div>
            <p className="text-danger">{selectors}</p>
            <div className="alert alert-warning">
              Solomon cannot plot graph for some selectors.
            </div>
          </div>
        );
      } else {
        selectorsEl = (
          <div>
            <p className="text-danger">{selectors}</p>
            <div className="alert alert-warning">
              For now, Solomon cannot plot graph for some selectors.
            </div>
          </div>
        );
      }

      mainBlock = (
        <div>
          <LabeledValue label="Selectors" value={selectorsEl} />
          {predicateContents}
        </div>
      );
      break;
    }
    case ALERT_TYPES.EXPRESSION: {
      const alertContent = isTemplated ? alert.templateData.type.expression : alert.type.expression;

      const trimmedProgram = (alertContent.program || '').trim();

      const programContent = trimmedProgram
        ? (
          <AlertProgram
            projectId={alert.projectId}
            code={trimmedProgram}
          />
        )
        : null;

      const checkExpression = alertContent.checkExpression
        ? (
          <AlertProgram
            projectId={alert.projectId}
            code={alertContent.checkExpression}
          />
        )
        : null;

      if (!programContent && !checkExpression) {
        mainBlock = null;
        break;
      }

      mainBlock = (
        <div>
          {trimmedProgram ? <LabeledValue label="Program" value={programContent} /> : null}
          {checkExpression ? <LabeledValue label="Check expression" value={checkExpression} /> : null}
        </div>
      );
      break;
    }
    default:
      mainBlock = null;
  }

  return mainBlock;
};

MainBlock.propTypes = {
  alert: PropTypes.object.isRequired,
  alertType: PropTypes.string.isRequired,
  isMultiAlert: PropTypes.bool.isRequired,
};

const getPolicyLabelByValue = (value, options) => {
  const filteredOptions = options.filter((o) => o.value === value || o.value === String(value));

  const style = (filteredOptions.length !== 1) ? 'default' : filteredOptions[0].style;
  const title = (filteredOptions.length !== 1) ? '(unknown)' : filteredOptions[0].title;

  return (<span className={`label label-${style}`}>{title}</span>);
};

const GenericOptionsPanel = ({ alert }) => {
  const alertType = getAlertType(alert.type, alert);

  const isMultiAlert = !isEmpty(alert.groupByLabels);
  const alertTypeTitle = alertType === ALERT_TYPES.FROM_TEMPLATE
    ? getAlertTypeTitle(getAlertType(alert.templateData.type, alert), isMultiAlert)
    : getAlertTypeTitle(alertType, isMultiAlert);

  let groupByLabelsBlock = null;

  if (alert.groupByLabels) {
    groupByLabelsBlock = (
      <LabeledValue
        label="Group by labels"
        value={<LabelsList values={alert.groupByLabels} />}
      />
    );
  }

  let resolvedEmptyPolicyBlock = null;
  let noPointsPolicyBlock = null;

  if (alert.resolvedEmptyPolicy && alert.resolvedEmptyPolicy !== RESOLVED_EMPTY_POLICIES.DEFAULT) {
    resolvedEmptyPolicyBlock = (
      <LabeledValue
        label="No metrics policy"
        value={getPolicyLabelByValue(alert.resolvedEmptyPolicy, RESOLVED_EMPTY_POLICY_OPTIONS)}
      />
    );
  }

  if (alert.noPointsPolicy && alert.noPointsPolicy !== NO_POINTS_POLICIES.DEFAULT) {
    noPointsPolicyBlock = (
      <LabeledValue
        label="No points policy"
        value={getPolicyLabelByValue(alert.noPointsPolicy, NO_POINTS_POLICY_OPTIONS)}
      />
    );
  }

  const isTemplate = Boolean(alert.type.fromTemplate || alert.fromTemplate);

  return (
    <Panel title="Generic options">
      <LabeledValue label="Id" value={alert.id} />
      {isTemplate && <LabeledValue label="Service provider" value={alert.templateData.serviceProviderId} />}
      <DescriptionBlock alert={alert} />
      <LabeledValue label="Type" value={alertTypeTitle} />
      {isTemplate && <LabeledValue label="Template" value={alert.templateData.id} />}
      {isTemplate && <LabeledValue label="Template version" value={alert.templateData.templateVersionTag} />}
      {groupByLabelsBlock}
      <DurationDelayBlocks alert={alert} />
      {<MainBlock alert={alert} alertType={alertType} isMultiAlert={isMultiAlert} />}
      {resolvedEmptyPolicyBlock}
      {noPointsPolicyBlock}
      <ChannelsBlock alert={alert} />
      <AnnotationsBlock annotations={alert.annotations} label="Annotations" />
      <AnnotationsBlock annotations={alert.serviceProviderAnnotations} label="Service provider annotations" />
      <EntityInfo entity={alert} />
    </Panel>
  );
};

GenericOptionsPanel.propTypes = {
  alert: PropTypes.object.isRequired,
};

export default GenericOptionsPanel;
