import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import isEmpty from 'lodash/isEmpty';
import entries from 'lodash/entries';
import { Link } from 'react-router-dom';

import Breadcrumb from '../../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../../components/PageTitle';

import { formatSearch, parseSearch } from '../../../utils/url';
import AlertSinglestat from './AlertSinglestat';
import AnnotationSuggest from '../components/AnnotationSuggest/AnnotationSuggest';

import { loadAlert, clearAlert } from '../../../store/reducers/alerts/alert';
import { loadAlertEvalState } from '../../../store/reducers/alerts/alertEvalState';
import { clearSubAlert, loadSubAlert } from '../../../store/reducers/alerts/subAlert';
import { loadSubAlertEvalState } from '../../../store/reducers/alerts/subAlertEvalState';

const AUTOREFRESH_DELAY_MILLIS = 30000;

class AlertSingleStatPage extends PureComponent {
  static mapSearchToState(search) {
    const params = parseSearch(search);

    const annotationKeys = params.get('annotationKeys');

    return {
      annotationKeys: annotationKeys ? annotationKeys.split(',') : [],
    };
  }

  static mapStateToSearch(state) {
    const search = {};

    if (state.annotationKeys.length > 0) {
      search.annotationKeys = state.annotationKeys.join(',');
    }

    return `?${formatSearch(search)}`;
  }

  static mapStateToParams(state) {
    const params = {};

    if (state.annotationKeys.length > 0) {
      params.annotationKeys = state.annotationKeys.join(',');
    }

    return params;
  }

  constructor(props) {
    super(props);
    this.state = AlertSingleStatPage.mapSearchToState(props.location.search);
  }

  componentDidMount() {
    this.loadAlertWithState(this.props.match.params);
  }

  componentDidUpdate(prevProps) {
    const { params } = this.props.match;
    const prevParams = prevProps.match.params;

    if (params.projectId !== prevParams.projectId
      || params.alertId !== prevParams.alertId) {
      this.clearAlertAutorefresh();
      this.loadAlertWithState(params);
    }
  }

  componentWillUnmount() {
    this.clearAlertAutorefresh();
    this.props.clearAlert();
    this.props.clearSubAlert();
  }

  loadAlertWithState = (params) => {
    const { projectId, alertId, subAlertId } = params;

    const promise = subAlertId
      ? this.props.loadSubAlert(projectId, alertId, subAlertId)
      : this.props.loadAlert(projectId, alertId);

    promise.then(() => this.runAlertAutorefresh());
  };

  loadAlertEvalState = () => {
    const { projectId, alertId, subAlertId } = this.props.match.params;

    const evalParams = AlertSingleStatPage.mapStateToParams(this.state);

    if (subAlertId) {
      this.props.loadSubAlertEvalState(projectId, alertId, subAlertId, evalParams);
    } else {
      this.props.loadAlertEvalState(projectId, alertId, evalParams);
    }
  };

  runAlertAutorefresh = () => {
    try {
      this.loadAlertEvalState();
    } catch (e) {
      console.error('failed to refresh alert singlestat', e);
    }
    this._autorefreshHandler = window.setTimeout(() => {
      this.runAlertAutorefresh();
    }, AUTOREFRESH_DELAY_MILLIS);
  };

  clearAlertAutorefresh = () => {
    if (this._autorefreshHandler != null) {
      window.clearTimeout(this._autorefreshHandler);
    }
  };

  changeAnnotationKeys = (values) => {
    const newAnnotationKeys = values.map((v) => v.value);
    this.setState({ annotationKeys: newAnnotationKeys }, () => this.reloadSubAlerts());
  };

  changeUrl = () => {
    this.props.history.replace(AlertSingleStatPage.mapStateToSearch(this.state));
  };

  reloadSubAlerts = () => {
    this.changeUrl();
    this.loadAlertEvalState();
  };

  render() {
    const {
      parentAlert, subAlert,
      parentAlertEvalState, subAlertEvalState,
      location, match,
    } = this.props;

    const { projectId, alertId, subAlertId } = match.params;

    const fullscreen = location.pathname.endsWith('/fullscreen') || location.pathname.endsWith('/fullscreen/');

    const alert = subAlertId ? subAlert : parentAlert;
    const evalState = subAlertId ? subAlertEvalState : parentAlertEvalState;

    if (isEmpty(alert)) {
      return <div>Loading...</div>;
    }

    let body;

    if (!isEmpty(alert.groupByLabels)) {
      body = (
        <div className="alert alert-danger">
          This alert must not support sub alerts
        </div>
      );
    } else if (isEmpty(evalState)) {
      body = <div>Loading...</div>;
    } else {
      const alertAnnotations = subAlertId ? alert.parent.annotations : alert.annotations;

      const alertAnnotationKeys = Object.keys(alertAnnotations || {});

      const singlestat = (
        <AlertSinglestat
          projectId={projectId}
          alertId={alertId}
          evalState={evalState}
          annotationKeys={this.state.annotationKeys}
          isTargetBlank={fullscreen}
        />
      );

      if (fullscreen) {
        return singlestat;
      }

      const fullscreenUrl = subAlertId
        ? `/admin/projects/${projectId}/alerts/${alertId}/subAlerts/${subAlertId}/singlestat/fullscreen${this.props.location.search}`
        : `/admin/projects/${projectId}/alerts/${alertId}/singlestat/fullscreen${this.props.location.search}`;

      body = (
        <div>
          <div className="row">
            <div className="btn-toolbar">
              <AnnotationSuggest
                annotations={alertAnnotationKeys}
                selectedValues={this.state.annotationKeys}
                onChange={this.changeAnnotationKeys}
              />
              <div className="btn-group">
                <Link className="btn btn-default" to={fullscreenUrl}>
                  <i className="glyphicon glyphicon-fullscreen" />
                </Link>
              </div>
            </div>
          </div>
          <div className="row" style={{ marginTop: '5px' }}>
            <div>
              {singlestat}
            </div>
          </div>
        </div>
      );
    }

    const labelValues = isEmpty(alert.labels)
      ? alert.id
      : entries(alert.labels).map((entry) => `${entry[0]}=${entry[1]}`).join(', ');

    const alertName = subAlertId ? subAlert.parent.name : alert.name;

    const title = subAlertId ? `Singlestat of sub alert ${labelValues}` : `Singlestat of alert ${alertName}`;

    if (fullscreen) {
      return body;
    }

    return (
      <div>
        <Breadcrumb match={this.props.match} alertName={alertName} />
        <PageTitle title={title} />
        {body}
      </div>
    );
  }
}

AlertSingleStatPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  parentAlert: PropTypes.object.isRequired,
  subAlert: PropTypes.object.isRequired,
  parentAlertEvalState: PropTypes.object.isRequired,
  subAlertEvalState: PropTypes.object.isRequired,
  loadAlert: PropTypes.func.isRequired,
  loadAlertEvalState: PropTypes.func.isRequired,
  loadSubAlert: PropTypes.func.isRequired,
  loadSubAlertEvalState: PropTypes.func.isRequired,
  clearAlert: PropTypes.func.isRequired,
  clearSubAlert: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  parentAlert: state.alert,
  subAlert: state.subAlert,
  parentAlertEvalState: state.alertEvalState,
  subAlertEvalState: state.subAlertEvalState,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadAlert,
  loadAlertEvalState,
  loadSubAlert,
  loadSubAlertEvalState,
  clearAlert,
  clearSubAlert,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(AlertSingleStatPage);

export default connectedPage;
