import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import isEmpty from 'lodash/isEmpty';

import { formatSearch, parseSearch } from '../../../utils/url';
import AlertSinglestat from './AlertSinglestat';
import {
  fetchAlert,
  fetchAlertEvalState,
  fetchSubAlert,
  fetchSubAlertEvaluationState,
} from '../../../api/alerts';

const AUTOREFRESH_DELAY_MILLIS = 30000;

class AlertSinglestatWidget extends PureComponent {
  static mapSearchToState(search) {
    const params = parseSearch(search);

    const annotationKeys = params.get('annotationKeys');

    return {
      annotationKeys: annotationKeys ? annotationKeys.split(',') : [],
    };
  }

  static mapStateToSearch(state) {
    const search = {};

    if (state.annotationKeys.length > 0) {
      search.annotationKeys = state.annotationKeys.join(',');
    }

    return `?${formatSearch(search)}`;
  }

  static mapStateToParams(state) {
    const params = {};

    if (state.annotationKeys.length > 0) {
      params.annotationKeys = state.annotationKeys.join(',');
    }

    return params;
  }

  constructor(props) {
    super(props);
    this.state = AlertSinglestatWidget.mapSearchToState(props.search);
  }

  componentDidMount() {
    this.loadAlertWithState(this.props.match.params);
  }

  componentDidUpdate(prevProps) {
    const { params } = this.props.match;
    const prevParams = prevProps.match.params;

    if (params.projectId !== prevParams.projectId
      || params.alertId !== prevParams.alertId) {
      this.clearAlertAutorefresh();
      this.loadAlertWithState(params);
    }
  }

  componentWillUnmount() {
    this.clearAlertAutorefresh();
  }

  loadAlertWithState = (params) => {
    const { projectId, alertId, subAlertId } = params;

    const promise = subAlertId
      ? fetchSubAlert(projectId, alertId, subAlertId).then((resp) => {
        this.setState({ subAlert: resp });
      })
      : fetchAlert(projectId, alertId).then((resp) => {
        this.setState({ parentAlert: resp });
      });

    promise.then(() => this.runAlertAutorefresh());
  };

  loadAlertEvalState = async () => {
    const { projectId, alertId, subAlertId } = this.props.match.params;

    const evalParams = AlertSinglestatWidget.mapStateToParams(this.state);

    if (subAlertId) {
      const subAlertEvalState = await fetchSubAlertEvaluationState(
        projectId,
        alertId,
        subAlertId,
        evalParams,
      );
      this.setState({ subAlertEvalState });
    } else {
      const parentAlertEvalState = await fetchAlertEvalState(projectId, alertId, evalParams);
      this.setState({ parentAlertEvalState });
    }
  };

  runAlertAutorefresh = () => {
    try {
      this.loadAlertEvalState();
    } catch (e) {
      console.error('failed to refresh alert singlestat', e);
    }
    this._autorefreshHandler = window.setTimeout(() => {
      this.runAlertAutorefresh();
    }, AUTOREFRESH_DELAY_MILLIS);
  };

  clearAlertAutorefresh = () => {
    if (this._autorefreshHandler != null) {
      window.clearTimeout(this._autorefreshHandler);
    }
  };

  render() {
    const {
      parentAlert, subAlert,
      parentAlertEvalState, subAlertEvalState,
    } = this.state;

    const { match, element } = this.props;
    const { projectId, alertId, subAlertId } = match.params;

    const alert = subAlertId ? subAlert : parentAlert;

    const evalState = subAlertId ? subAlertEvalState : parentAlertEvalState;

    if (isEmpty(alert)) {
      return <div style={{ textAlign: 'center' }}>Loading alert...</div>;
    }

    let body;

    if (!isEmpty(alert.groupByLabels)) {
      body = (
        <div className="alert alert-danger">
          This alert must not support sub alerts
        </div>
      );
    } else if (isEmpty(evalState)) {
      body = <div style={{ textAlign: 'center' }}>Loading evaluation...</div>;
    } else {
      const alertAnnotations = subAlertId ? alert.parent.annotations : alert.annotations;
      Object.keys(alertAnnotations || {});

      return (
        <AlertSinglestat
          projectId={projectId}
          alertId={alertId}
          evalState={evalState}
          annotationKeys={this.state.annotationKeys}
          element={element}
          isTargetBlank
        />
      );
    }

    return body;
  }
}

AlertSinglestatWidget.propTypes = {
  element: PropTypes.element.isRequired,
  match: PropTypes.object.isRequired,
  search: PropTypes.string.isRequired,
};

export default AlertSinglestatWidget;
