import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';

import isEmpty from 'lodash/isEmpty';

import { formatSearch, parseSearch } from '../../../utils/url';
import PageTitle from '../../../components/PageTitle';
import { clearAlertStates, loadAlertStates } from '../../../store/reducers/alerts/alertStates';
import AlertSuggest from './AlertSuggest';
import mapAlertEvalStatesToTiles from './utils';
import FullscreenFlexableTiles from '../../../components/Tiles/FullscreenFlexableTiles';
import ButtonGroup from '../../../components/ButtonGroup';
import Breadcrumb from '../../../components/Breadcrumb/Breadcrumb';
import { wrapSubProjectReadPage } from '../../projects/wrapSubProjectPage';

const AUTOREFRESH_DELAY_MILLIS = 30000;

class AlertStatesPage extends PureComponent {
  static mapSearchToState(search) {
    const params = parseSearch(search);
    const alertIdsParam = params.get('alertIds') || '';
    const alertIds = alertIdsParam ? alertIdsParam.split(',') : [];

    const filterByEvalStatusParam = params.get('filterByEvalStatus') || '';
    const filterByEvalStatus = filterByEvalStatusParam ? filterByEvalStatusParam.split(',') : [];

    return { alertIds, filterByEvalStatus };
  }

  static mapStateToSearch(state) {
    const search = {};

    if (!isEmpty(state.alertIds)) {
      search.alertIds = state.alertIds.join(',');
    }

    if (!isEmpty(state.filterByEvalStatus)) {
      search.filterByEvalStatus = state.filterByEvalStatus.join(',');
    }

    return `?${formatSearch(search)}`;
  }

  constructor(props) {
    super(props);
    this.state = AlertStatesPage.mapSearchToState(props.location.search);
  }

  componentDidMount() {
    if (!isEmpty(this.state.alertIds)) {
      this.runAutorefresh();
    }
  }

  componentWillUnmount() {
    this.clearAutorefresh();
    this.props.clearAlertStates();
  }

  handleAlertIdsChange = (alertIds) => {
    this.setState({ alertIds }, () => this.reloadAlertStates());
  };

  handleEvalStatusesChange = (filterByEvalStatus) => {
    this.setState({ filterByEvalStatus }, () => this.reloadAlertStates());
  };

  reloadAlertStates = () => {
    this.props.history.replace(AlertStatesPage.mapStateToSearch(this.state));
    this.runAutorefresh();
  };

  loadAlertStates = () => {
    const { alertIds, filterByEvalStatus } = this.state;

    if (isEmpty(alertIds)) {
      this.props.clearAlertStates();
    } else {
      const { projectId } = this.props.match.params;
      this.props.loadAlertStates(projectId, alertIds, filterByEvalStatus);
    }
  };

  clearAutorefresh = () => {
    if (this._autorefreshHandler) {
      window.clearTimeout(this._autorefreshHandler);
    }
    this._autorefreshHandler = null;
  };

  runAutorefresh = () => {
    try {
      this.clearAutorefresh();
      this.loadAlertStates();
    } catch (e) {
      console.error('failed to refresh alert states', e);
    }

    if (isEmpty(this.state.alertIds)) {
      this._autorefreshHandler = null;
    } else {
      this._autorefreshHandler = window.setTimeout(() => {
        this.runAutorefresh();
      }, AUTOREFRESH_DELAY_MILLIS);
    }
  };

  render() {
    const { loading, data, error } = this.props;

    const { pathname } = this.props.location;

    const fullscreen = pathname.endsWith('/fullscreen') || pathname.endsWith('/fullscreen/');

    const { projectId } = this.props.match.params;

    let tilesEl = null;

    if (loading && data === null) {
      tilesEl = <div>Loading...</div>;
    } else if (error) {
      tilesEl = <div>Failed to load data</div>;
    } else if (data !== null) {
      const tiles = mapAlertEvalStatesToTiles(data);

      tilesEl = <FullscreenFlexableTiles tiles={tiles} fullscreen={fullscreen} />;
    } else {
      tilesEl = <div>No items</div>;
    }

    if (fullscreen) {
      return tilesEl;
    }

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title="Alert states" />
        <div>
          <table style={{ border: '0' }}>
            <tbody>
              <tr>
                <td style={{ width: '100%' }}>
                  <AlertSuggest
                    projectId={projectId}
                    values={this.state.alertIds}
                    onChange={this.handleAlertIdsChange}
                  />
                </td>
                <td style={{ padding: '5px', width: '0%' }}>
                  <ButtonGroup
                    choices={['OK', 'WARN', 'ALARM', 'NO_DATA', 'ERROR']}
                    labels={['OK', 'Warning', 'Alarm', 'No data', 'Error']}
                    values={this.state.filterByEvalStatus}
                    onChange={this.handleEvalStatusesChange}
                    style={{ display: 'flex' }}
                  />
                </td>
                <td style={{ padding: '5px', width: '0%' }}>
                  <Link
                    className="btn btn-default"
                    to={`/admin/projects/${projectId}/alertStates/fullscreen${this.props.location.search}`}
                  >
                    <i className="glyphicon glyphicon-fullscreen" />
                  </Link>
                </td>
              </tr>
            </tbody>
          </table>
        </div>
        <div style={{ marginTop: '5px' }}>
          {tilesEl}
        </div>
      </div>
    );
  }
}

AlertStatesPage.propTypes = {
  location: PropTypes.object.isRequired,
  match: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  loading: PropTypes.bool.isRequired,
  data: PropTypes.array,
  error: PropTypes.object,
  loadAlertStates: PropTypes.func.isRequired,
  clearAlertStates: PropTypes.func.isRequired,
};

AlertStatesPage.defaultProps = {
  data: null,
  error: null,
};

const mapStateToProps = (state) => state.alertStates;

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadAlertStates,
  clearAlertStates,
}, dispatch);

export default wrapSubProjectReadPage(
  connect(mapStateToProps, mapDispatchToProps)(AlertStatesPage),
);
