import React, { PureComponent } from 'react';
import { connect } from 'react-redux';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';
import { bindActionCreators } from 'redux';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import LoadMore from '../../components/LoadMore/LoadMore';

import {
  ALERT_EVAL_STATUSES,
  ALERT_STATE_OPTIONS,
  ALERT_TYPE_OPTIONS,
  NOTIFICATION_STATUSES,
  getAlertTypeTitle,
  parseAlertType,
} from './constants';

import { loadNextProjectAlerts, loadProjectAlerts } from '../../store/reducers/alerts/alertsPage';
import { formatSearch, parseSearch } from '../../utils/url';
import ClickableNotificationStats from './components/ClickableNotificationStats';
import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';
import { PROJECT_PERMISSIONS } from '../../auth/ProjectPermissions';
import { loadServiceProviders } from '../../store/reducers/serviceProviders/serviceProviders';

class AlertsPage extends PureComponent {
  static mapSearchToState = (search) => {
    const parsedSearch = parseSearch(search);
    const nameFilter = parsedSearch.get('text') || '';
    const stateFilter = parsedSearch.get('state') || 'ALL';
    const typeFilter = parsedSearch.get('type') || 'ALL';
    const evalStatusFilter = parsedSearch.get('evalStatus') || 'ALL';
    const notificationStatusFilter = parsedSearch.get('notificationStatus') || 'ALL';
    const pageSize = parsedSearch.get('pageSize') || '30';
    const sourceFilter = parsedSearch.get('templateServiceProviderId') || 'ALL';

    return {
      stateFilter,
      nameFilter,
      typeFilter,
      evalStatusFilter,
      notificationStatusFilter,
      pageSize,
      sourceFilter,
    };
  };

  static mapStateToParams = (state) => {
    const params = {};
    if (state.nameFilter) {
      params.filterByName = state.nameFilter;
    }
    if (state.typeFilter !== 'ALL') {
      params.filterByTypes = state.typeFilter;
    }
    if (state.evalStatusFilter !== 'ALL') {
      params.filterByEvaluationStatus = state.evalStatusFilter;
    }
    if (state.notificationStatusFilter !== 'ALL') {
      params.filterByNotificationStatus = state.notificationStatusFilter;
    }
    if (state.stateFilter !== 'ALL') {
      params.filterByStates = state.stateFilter;
    }
    if (state.sourceFilter !== 'ALL') {
      params.templateServiceProviderId = state.sourceFilter;
    }
    if (state.pageSize) {
      params.pageSize = state.pageSize;
    } else {
      params.pageSize = '30';
    }
    return params;
  };

  static mapStateToSearch = (state) => {
    const params = {};
    if (state.nameFilter) {
      params.text = state.nameFilter;
    }
    if (state.typeFilter !== 'ALL') {
      params.type = state.typeFilter;
    }
    if (state.evalStatusFilter !== 'ALL') {
      params.evalStatus = state.evalStatusFilter;
    }
    if (state.notificationStatusFilter !== 'ALL') {
      params.notificationStatus = state.notificationStatusFilter;
    }
    if (state.stateFilter !== 'ALL') {
      params.state = state.stateFilter;
    }
    if (state.pageSize !== '30') {
      params.pageSize = state.pageSize;
    }

    if (state.sourceFilter) {
      params.templateServiceProviderId = state.sourceFilter;
    }
    return `?${formatSearch(params)}`;
  };

  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
    this.state = AlertsPage.mapSearchToState(props.location.search);
  }

  componentDidMount() {
    this.inputEl.focus();
    this.props.loadServiceProviders();
    this.props.loadProjectAlerts(this._projectId, AlertsPage.mapStateToParams(this.state));
  }

  componentDidUpdate(prevProps) {
    const { projectId } = this.props.match.params;
    const prevProjectId = prevProps.match.params.projectId;
    if (projectId !== prevProjectId
      || this.props.location.search !== prevProps.location.search) {
      this._projectId = projectId;

      const newState = AlertsPage.mapSearchToState(this.props.location.search);

      this.setState(newState);
      this.props.loadProjectAlerts(this._projectId, AlertsPage.mapStateToParams(newState));
    }
  }

  onAlertFilterChange = (event) => {
    event.preventDefault();
    this.setState({ [event.target.name]: event.target.value }, () => this.reloadAlerts());
  };

  onLoadMore = () => {
    const pageToken = this.props.alertsPage.nextPageToken;
    const newParams = { ...AlertsPage.mapStateToParams(this.state), pageToken, pageSize: 30 };
    this.props.loadNextProjectAlerts(this._projectId, newParams);
  };

  reloadAlerts = () => {
    const search = AlertsPage.mapStateToSearch(this.state);
    this.props.history.replace(search);
  };

  render() {
    const { alertsPage, projectAuth, serviceProviders } = this.props;
    const alerts = alertsPage.items || [];

    const canCreate = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_UPDATE);
    const serviceProvidersList = serviceProviders?.data?.serviceProviders || [];

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Alerts of ${this.props.projectName}`} />
        <div className="btn-toolbar table-toolbar">
          {canCreate && (
            <div className="btn-group">
              <Link to={`/admin/projects/${this._projectId}/alerts/choose`} className="btn btn-outline btn-success">
                <i className="glyphicon glyphicon-plus" />
                {' '}
                Create New
              </Link>
            </div>
          )}
        </div>
        <table className="table table-condensed table-hover">
          <thead>
            <tr>
              <th>Name</th>
              <th>Type</th>
              <th>Service provider</th>
              <th colSpan={ALERT_EVAL_STATUSES.length}>Evaluation stats</th>
              <th>Notification stats</th>
              <th>State</th>
            </tr>
          </thead>
          <tbody>
            <tr>
              <td>
                <input
                  className="form-control"
                  name="nameFilter"
                  value={this.state.nameFilter}
                  placeholder="Type to filter alert names"
                  onChange={this.onAlertFilterChange}
                  ref={(el) => { this.inputEl = el; }}
                />
              </td>
              <td>
                <select
                  className="form-control"
                  name="typeFilter"
                  value={this.state.typeFilter}
                  onChange={this.onAlertFilterChange}
                >
                  <option value="ALL">Any type</option>
                  {ALERT_TYPE_OPTIONS.map((option) => (
                    <option key={option.upperValue} value={option.upperValue}>
                      {option.title}
                    </option>
                  ))}
                </select>
              </td>
              <td>
                <select
                  className="form-control"
                  name="sourceFilter"
                  value={this.state.sourceFilter}
                  onChange={this.onAlertFilterChange}
                >
                  <option value="ALL">Any provider</option>
                  <option value="-">Custom</option>
                  {serviceProvidersList.map((option) => (
                    <option key={option.upperValue} value={option.upperValue}>
                      {option.id}
                    </option>
                  ))}
                </select>
              </td>
              <td colSpan={ALERT_EVAL_STATUSES.length}>
                <select
                  className="form-control"
                  name="evalStatusFilter"
                  value={this.state.evalStatusFilter}
                  onChange={this.onAlertFilterChange}
                >
                  <option value="ALL">Any status</option>
                  {ALERT_EVAL_STATUSES.map((option) => (
                    <option key={option.value} value={option.value}>
                      {option.title}
                    </option>
                  ))}
                </select>
              </td>
              <td>
                <select
                  className="form-control"
                  name="notificationStatusFilter"
                  value={this.state.notificationStatusFilter}
                  onChange={this.onAlertFilterChange}
                >
                  <option value="ALL">Any status</option>
                  {NOTIFICATION_STATUSES.map((option) => (
                    <option key={option.value} value={option.value}>
                      {option.title}
                    </option>
                  ))}
                </select>
              </td>
              <td>
                <select
                  className="form-control"
                  name="stateFilter"
                  value={this.state.stateFilter}
                  onChange={this.onAlertFilterChange}
                >
                  {ALERT_STATE_OPTIONS.map((option) => (
                    <option key={option.value} value={option.value}>
                      {option.title}
                    </option>
                  ))}
                </select>
              </td>
            </tr>
            {alerts.map((alert, i) => (
              <tr key={alert.id}>
                <td>
                  <Link to={`/admin/projects/${this._projectId}/alerts/${alert.id}`} tabIndex={i + 1}>
                    {alert.name}
                  </Link>
                </td>
                <td>
                  {getAlertTypeTitle(parseAlertType(alert.type, alert), alert.multiAlert)}
                </td>
                <td>
                  {alert.type === 'FROM_TEMPLATE' ? alert.typeData.fromTemplate.templateServiceProviderId : 'Custom'}
                </td>
                {ALERT_EVAL_STATUSES.map((evalStatus) => {
                  const count = alert.evaluationStats[evalStatus.statsName];

                  let evalContent = null;

                  if (count) {
                    evalContent = (
                      <span className={`label label-${evalStatus.style}`}>
                        {count}
                      </span>
                    );

                    if (alert.multiAlert) {
                      const subAlertsUrl = `/admin/projects/${this._projectId}/alerts/${alert.id}/subAlerts?filterByEvalStatus=${evalStatus.value}`;
                      evalContent = (
                        <Link to={subAlertsUrl}>
                          {evalContent}
                        </Link>
                      );
                    } else {
                      const alertUrl = `/admin/projects/${this._projectId}/alerts/${alert.id}`;

                      evalContent = (
                        <Link to={alertUrl}>
                          {evalContent}
                        </Link>
                      );
                    }
                  }

                  return <td key={evalStatus.value}>{evalContent}</td>;
                })}
                <td>
                  <ClickableNotificationStats
                    projectId={this._projectId}
                    alertId={alert.id}
                    multiAlert={alert.multiAlert}
                    stats={alert.notificationStats}
                  />
                </td>
                <td>
                  {alert.state === 'MUTED' ? <i className="glyphicon glyphicon-volume-off" /> : null}
                </td>
              </tr>
            ))}
          </tbody>
        </table>
        {alertsPage.nextPageToken ? <LoadMore onLoadMore={this.onLoadMore} /> : null}
      </div>
    );
  }
}

AlertsPage.propTypes = {
  match: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  projectName: PropTypes.string.isRequired,
  alertsPage: PropTypes.object.isRequired,
  projectAuth: PropTypes.object.isRequired,
  loadProjectAlerts: PropTypes.func.isRequired,
  loadNextProjectAlerts: PropTypes.func.isRequired,
  loadServiceProviders: PropTypes.func.isRequired,
  serviceProviders: PropTypes.object.isRequired,
};

const mapStateToProps = (state) => ({
  projectName: state.projectData.project.name,
  alertsPage: state.alertsPage,
  serviceProviders: state.serviceProvidersPage,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadProjectAlerts,
  loadNextProjectAlerts,
  loadServiceProviders,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(AlertsPage);

export default wrapSubProjectReadPage(connectedPage);
