/* eslint-disable max-classes-per-file */
import React, { PureComponent } from 'react';
import { connect } from 'react-redux';
import PropTypes from 'prop-types';
import { bindActionCreators } from 'redux';

import isEmpty from 'lodash/isEmpty';
import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';

import { loadDefaultTemplates } from '../../store/reducers/alerts/alertsTemplates';
import { loadServiceProviders } from '../../store/reducers/serviceProviders/serviceProviders';
import { formatSearch, parseSearch } from '../../utils/url';
import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';
import FormSelect from '../../components/forms/FormSelect';
import TableViewer from '../../components/TableViewer';
import FormElement from '../../components/forms/FormElement';
import FormArea from '../../components/forms/FormArea';
import { inputTargetValue } from '../../utils/inputs';
import FormButtons from '../../components/forms/FormButtons';
import { saveAlerts } from '../../store/reducers/alerts/alert';
import ChannelListEditor from './AlertEditPage/ChannelListEditor/ChannelListEditor';
import InternalOrExternalLink from '../../components/InternalOrExternalLink';
import { ALERT_EVAL_STATUS } from './constants';

class TemplateRow extends PureComponent {
  render() {
    const { value } = this.props;
    const type = value.labels?.resourceType;
    return (
      <tr>
        <td>{value.name || ''}</td>
        <td>{value.description || ''}</td>
        <td>{type || 'all'}</td>
      </tr>
    );
  }
}

TemplateRow.propTypes = {
  value: PropTypes.object.isRequired,
};

class CreationResultRow extends PureComponent {
  render() {
    const { value } = this.props;
    const href = `/admin/projects/${value.projectId}/alerts/${value.alertId || ''}`;
    return (
      <tr onClick>
        <td>
          <InternalOrExternalLink target="_blank" href={href}>
            {value.resourceRow}
          </InternalOrExternalLink>
        </td>
        <td>{value.templateId || ''}</td>
      </tr>
    );
  }
}

CreationResultRow.propTypes = {
  value: PropTypes.object.isRequired,
};

class AlertsTemplates extends PureComponent {
  static mapSearchToState = (search, resourceIdsParam, channelsParam) => {
    const parsedSearch = parseSearch(search);
    const serviceProviderId = parsedSearch.get('serviceProviderId') || 'monitoring';
    const pageSize = parsedSearch.get('pageSize') || '100';
    const format = parsedSearch.get('format') || 'new';
    const templateIds = [];
    const channels = channelsParam || [];
    const creationResult = {};
    const explainFormat = '';
    const resourcesParam = parsedSearch.getAll('r');
    const resourcesStr = resourcesParam
      ? resourcesParam.join('\n')
      : '';
    const resourceIds = resourceIdsParam || resourcesStr;

    return {
      serviceProviderId,
      pageSize,
      templateIds,
      resourceIds,
      explainFormat,
      channels,
      creationResult,
      format,
    };
  };

  static mapStateToParams = (state) => {
    const params = {};
    params.serviceProviderId = state.serviceProviderId;
    if (state.pageSize) {
      params.pageSize = state.pageSize;
    } else {
      params.pageSize = '100';
    }
    params.format = state.format;
    return params;
  };

  static mapStateToSearch = (state) => {
    const params = {};
    params.serviceProviderId = state.serviceProviderId;
    if (state.pageSize !== '100') {
      params.pageSize = state.pageSize;
    }
    params.format = state.format;
    return `?${formatSearch(params)}`;
  };

  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
    this.state = AlertsTemplates.mapSearchToState(props.location.search);
  }

  componentDidMount() {
    this.props.loadServiceProviders();
    this.props.loadDefaultTemplates(AlertsTemplates.mapStateToParams(this.state));
  }

  componentDidUpdate(prevProps) {
    const { projectId } = this.props.match.params;
    const prevProjectId = prevProps.match.params.projectId;
    if (projectId !== prevProjectId
      || this.props.location.search !== prevProps.location.search) {
      this._projectId = projectId;

      const newState = AlertsTemplates.mapSearchToState(
        this.props.location.search,
        this.state.resourceIds,
        this.state.channels,
      );
      this.setState(newState);
      this.props.loadDefaultTemplates(AlertsTemplates.mapStateToParams(newState));
    }
  }

  onAlertTemplateFilterChange = (event) => {
    event.preventDefault();
    this.setState({ [event.target.name]: event.target.value }, () => this.reloadTemplates());
  };

  reloadTemplates = () => {
    const search = AlertsTemplates.mapStateToSearch(this.state);
    this.props.history.replace(search);
  };

  onInputChange = (event) => {
    const { target } = event;
    const value = inputTargetValue(target);
    this.setState({ [target.name]: value });
  };

  onSubmit = (event) => {
    event.preventDefault();
    const { templatesPage } = this.props;
    const templateIds = (templatesPage.items || []).map((entry) => (entry.id));
    this.props.saveAlerts(this._projectId, {
      serviceProviderId: this.state.serviceProviderId,
      templateIds,
      channels: this.state.channels,
      rowsText: this.state.resourceIds,
      format: this.state.format,
    })
      .then((payload) => {
        if (payload) {
          this.setState({ creationResult: payload });
        }
      });
  };

  onChannelsChange = (channels) => {
    this.setState({ channels });
  };

  render() {
    const { templatesPage, serviceProviders } = this.props;
    const templates = templatesPage.items || [];
    const serviceProvidersList = serviceProviders?.data?.serviceProviders || [''];
    const serviceProviderOptions = serviceProvidersList.map((entry) => ({ title: entry.id, value: entry.id, style: 'default' }));
    const projectId = this._projectId;
    const creationResultAlerts = !isEmpty(this.state.creationResult.alerts)
      ? this.state.creationResult.alerts.map((entry) => ({
        projectId,
        ...entry,
      })) : [];

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title="Create alerts from templates for multiple resources" />
        <FormSelect
          name="serviceProviderId"
          label="Service provider"
          value={this.state.serviceProviderId}
          options={serviceProviderOptions}
          onChange={this.onAlertTemplateFilterChange}
          help="Service provider for alert templates"
        />
        <FormElement label="Templates" help="List of templates to create alerts">
          <TableViewer
            columns={['Name', 'Description', 'Resource type']}
            values={templates}
            row={TemplateRow}
            limit={10}
          />
        </FormElement>
        <FormElement
          label="Channels"
          help="Channels to notify alerts"
        >
          <ChannelListEditor
            projectId={this._projectId}
            channels={this.state.channels}
            notifyAboutStatusesDefault={
              [
                ALERT_EVAL_STATUS.ALARM.value,
                ALERT_EVAL_STATUS.WARN.value,
                ALERT_EVAL_STATUS.OK.value,
              ]
            }
            onChange={this.onChannelsChange}
          />
        </FormElement>
        <FormArea
          label="Resources"
          name="resourceIds"
          value={this.state.resourceIds}
          onChange={this.onInputChange}
          help="Resources in format: paramName:value1,paramName2:value2,resourceType:type"
          rows={10}
        />
        {!isEmpty(this.state.creationResult.alerts) && (
          <FormElement label="Created alerts" help="List of created alerts">
            <TableViewer
              columns={['Resource row Id', 'Template Id']}
              values={creationResultAlerts}
              row={CreationResultRow}
              limit={10}
            />
          </FormElement>
        )}
        {isEmpty(this.state.creationResult.alerts) && (
          <FormButtons onSubmit={this.onSubmit} />
        )}
      </div>
    );
  }
}

AlertsTemplates.propTypes = {
  match: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  templatesPage: PropTypes.object.isRequired,
  loadDefaultTemplates: PropTypes.func.isRequired,
  loadServiceProviders: PropTypes.func.isRequired,
  serviceProviders: PropTypes.object.isRequired,
  saveAlerts: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  templatesPage: state.alertsTemplatesPage,
  serviceProviders: state.serviceProvidersPage,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadServiceProviders,
  loadDefaultTemplates,
  saveAlerts,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(AlertsTemplates);

export default wrapSubProjectReadPage(connectedPage);
