import React, { PureComponent } from 'react';
import { connect } from 'react-redux';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';
import { bindActionCreators } from 'redux';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import LoadMore from '../../components/LoadMore/LoadMore';

import { loadNextTemplates, loadTemplates } from '../../store/reducers/alerts/alertsTemplates';
import { loadServiceProviders } from '../../store/reducers/serviceProviders/serviceProviders';
import { formatSearch, parseSearch } from '../../utils/url';
import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';

class AlertsTemplates extends PureComponent {
  static mapSearchToState = (search) => {
    const parsedSearch = parseSearch(search);
    const nameFilter = parsedSearch.get('name') || '';
    const serviceProviderIdFilter = parsedSearch.get('serviceProviderId') || 'ALL';
    const pageSize = parsedSearch.get('pageSize') || '30';

    return {
      nameFilter,
      serviceProviderIdFilter,
      pageSize,
    };
  };

  static mapStateToParams = (state) => {
    const params = {};
    if (state.nameFilter) {
      params.name = state.nameFilter;
    }
    if (state.serviceProviderIdFilter !== 'ALL') {
      params.serviceProviderId = state.serviceProviderIdFilter;
    }
    if (state.pageSize) {
      params.pageSize = state.pageSize;
    } else {
      params.pageSize = '30';
    }
    return params;
  };

  static mapStateToSearch = (state) => {
    const params = {};
    if (state.nameFilter) {
      params.name = state.nameFilter;
    }
    if (state.serviceProviderIdFilter !== 'ALL') {
      params.serviceProviderId = state.serviceProviderIdFilter;
    }
    if (state.pageSize !== '30') {
      params.pageSize = state.pageSize;
    }
    return `?${formatSearch(params)}`;
  };

  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
    this.state = AlertsTemplates.mapSearchToState(props.location.search);
  }

  componentDidMount() {
    this.inputEl.focus();
    this.props.loadServiceProviders();
    this.props.loadTemplates(AlertsTemplates.mapStateToParams(this.state));
  }

  componentDidUpdate(prevProps) {
    const { projectId } = this.props.match.params;
    const prevProjectId = prevProps.match.params.projectId;
    if (projectId !== prevProjectId
      || this.props.location.search !== prevProps.location.search) {
      this._projectId = projectId;

      const newState = AlertsTemplates.mapSearchToState(this.props.location.search);

      this.setState(newState);
      this.props.loadTemplates(AlertsTemplates.mapStateToParams(newState));
    }
  }

  onAlertFilterChange = (event) => {
    event.preventDefault();
    this.setState({ [event.target.name]: event.target.value }, () => this.reloadTemplates());
  };

  onLoadMore = () => {
    const pageToken = this.props.templatesPage.nextPageToken;
    const newParams = { ...AlertsTemplates.mapStateToParams(this.state), pageToken, pageSize: 30 };
    this.props.loadNextTemplates(newParams);
  };

  reloadTemplates = () => {
    const search = AlertsTemplates.mapStateToSearch(this.state);
    this.props.history.replace(search);
  };

  render() {
    const { templatesPage, serviceProviders } = this.props;
    const templates = templatesPage.items || [];
    const serviceProvidersList = serviceProviders?.data?.serviceProviders;

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Alert templates of ${this.props.projectName}`} />
        <table className="table table-condensed table-hover">
          <thead>
            <tr>
              <th>Name</th>
              <th>Service provider</th>
              <th>Description</th>
            </tr>
          </thead>
          <tbody>
            <tr>
              <td>
                <input
                  className="form-control"
                  name="nameFilter"
                  value={this.state.nameFilter}
                  placeholder="Type to filter template names"
                  onChange={this.onAlertFilterChange}
                  ref={(el) => { this.inputEl = el; }}
                />
              </td>
              <td>
                <select
                  className="form-control"
                  name="serviceProviderIdFilter"
                  value={this.state.serviceProviderIdFilter}
                  onChange={this.onAlertFilterChange}
                >
                  <option value="ALL">Any provider</option>
                  {(serviceProvidersList || []).map((option) => (
                    <option key={option.upperValue} value={option.upperValue}>
                      {option.id}
                    </option>
                  ))}
                </select>
              </td>
              <td />
            </tr>
            {templates.map((template) => (
              <tr key={template.id}>
                <td>
                  <Link to={`/admin/projects/${this._projectId}/alerts/new/?aType=fromTemplate&templateId=${template.id}&templateVersion=${template.templateVersionTag}`}>
                    {template.name}
                  </Link>
                </td>
                <td>
                  {template.serviceProviderId}
                </td>
                <td>
                  {template.description}
                </td>
              </tr>
            ))}
          </tbody>
        </table>
        {templatesPage.nextPageToken ? <LoadMore onLoadMore={this.onLoadMore} /> : null}
      </div>
    );
  }
}

AlertsTemplates.propTypes = {
  match: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  projectName: PropTypes.string.isRequired,
  templatesPage: PropTypes.object.isRequired,
  loadTemplates: PropTypes.func.isRequired,
  loadNextTemplates: PropTypes.func.isRequired,
  loadServiceProviders: PropTypes.func.isRequired,
  serviceProviders: PropTypes.object.isRequired,
};

const mapStateToProps = (state) => ({
  projectName: state.projectData.project.name,
  templatesPage: state.alertsTemplatesPage,
  serviceProviders: state.serviceProvidersPage,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadServiceProviders,
  loadTemplates,
  loadNextTemplates,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(AlertsTemplates);

export default wrapSubProjectReadPage(connectedPage);
