import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import { Link } from 'react-router-dom';
import isEmpty from 'lodash/isEmpty';
import entries from 'lodash/entries';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import AlertEvalInfo from './components/AlertEvalInfo';
import AlertNotificationInfo from './components/AlertNotificationInfo';

import SingleAlertStatus from './charts/SingleAlertStatus/SingleAlertStatus';

import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';
import { makeOldUiGraphUrl, mergeSelectorsAndLabels } from '../../utils/SelectorUtils';
import Selectors from '../../utils/Selectors';
import LabeledValue from '../../components/LabeledValue';
import SubAlertProgram from './SubAlertsPage/SubAlertProgram';
import Panel from '../../components/Panel';
import { ReadonlyTable as PredicateRulesView } from './components/PredicateRulesTable';
import AlertMutesPanel from './AlertPage/AlertMutesPanel';
import {
  AnnotationsBlock,
  ChannelsBlock, DescriptionBlock,
  DurationDelayBlocks,
} from './AlertPage/GenericOptionsPanel';
import { COMPARISON_OPTIONS, getTitleByValue, THRESHOLD_TYPE_OPTIONS } from './constants';

import { loadSubAlert, clearSubAlert } from '../../store/reducers/alerts/subAlert';
import { loadSubAlertEvalState, clearSubAlertEvalState } from '../../store/reducers/alerts/subAlertEvalState';
import { loadSubAlertNotificationState, clearSubAlertNotificationState } from '../../store/reducers/alerts/subAlertNotificationState';
import { loadExistingSubAlertExplanation } from '../../store/reducers/alerts/subAlertExplanation';

const AlertStatus = ({ status }) => {
  if (isEmpty(status)) {
    return <span>-</span>;
  }

  if (!isEmpty(status.descrption)) {
    return (
      <div className="alert alert-danger">
        <pre>{status.description}</pre>
      </div>
    );
  }

  return <span>{status.code}</span>;
};

AlertStatus.propTypes = {
  status: PropTypes.object,
};

AlertStatus.defaultProps = {
  status: {},
};

class SubAlertPage extends PureComponent {
  componentDidMount() {
    const { projectId, alertId, subAlertId } = this.props.match.params;
    this.loadSubAlertWithStats(projectId, alertId, subAlertId);
  }

  componentDidUpdate(prevProps) {
    if (this.props.match.params.subAlertId !== prevProps.match.params.subAlertId) {
      const { projectId, alertId, subAlertId } = this.props.match.params;
      this.loadSubAlertWithStats(projectId, alertId, subAlertId);
    }
  }

  componentWillUnmount() {
    this.props.clearSubAlert();
    this.props.clearSubAlertEvalState();
    this.props.clearSubAlertNotificationState();
  }

  handleLoadExplanation = (ts) => {
    const { projectId, alertId, subAlertId } = this.props.match.params;
    const time = new Date(ts).toISOString();
    this.props.loadExistingSubAlertExplanation(projectId, alertId, subAlertId, { time });
  };

  loadSubAlertWithStats(projectId, alertId, subAlertId) {
    this.props.loadSubAlert(projectId, alertId, subAlertId);
    this.props.loadSubAlertEvalState(projectId, alertId, subAlertId);
    this.props.loadSubAlertNotificationState(projectId, alertId, subAlertId);
    this.props.loadExistingSubAlertExplanation(projectId, alertId, subAlertId);
  }

  render() {
    const {
      subAlert,
      evalState,
      notificationState,
      subAlertExplanation,
    } = this.props;
    const { projectId, alertId, subAlertId } = this.props.match.params;

    if (isEmpty(subAlert)) {
      return <span>Loading...</span>;
    }

    const subAlertLabels = {};
    let subAlertSelector = '{';

    (subAlert.parent.groupByLabels || []).forEach((label) => {
      subAlertLabels[label] = subAlert.labels[label] || '-';
      subAlertSelector += `${label}="${subAlert.labels[label] || '-'}",`;
    });

    subAlertSelector = `${subAlertSelector.slice(0, -1)}}`;

    const labelValues = entries(subAlertLabels).map((entry) => `${entry[0]}=${entry[1]}`).join(', ');

    let oldUiGraphUrl = '';
    let genericOptionsContent = null;

    const { parent } = subAlert;

    if (parent.type.threshold) {
      const alertContent = parent.type.threshold;
      const aggregation = getTitleByValue(alertContent.timeAggregation, THRESHOLD_TYPE_OPTIONS);
      const comparison = getTitleByValue(alertContent.predicate, COMPARISON_OPTIONS);
      const predicateRulesList = alertContent.predicateRules || [];

      let predicateContents;
      if (predicateRulesList.length > 0) {
        const predicateRulesContent = (
          <PredicateRulesView
            predicateRules={predicateRulesList}
            limit={5}
          />
        );

        predicateContents = <LabeledValue label="Predicate rules" value={predicateRulesContent} />;
      } else {
        let predicateExpr;

        if (!aggregation || !comparison) {
          predicateExpr = '';
        } else {
          predicateExpr = `${aggregation} of points ${comparison} ${alertContent.threshold}`;
        }
        predicateContents = <LabeledValue label="Predicate" value={predicateExpr} />;
      }

      try {
        let parsedSelectors = Selectors.parse(parent.type.threshold.selectors);
        parsedSelectors = mergeSelectorsAndLabels(parsedSelectors, subAlertLabels);
        oldUiGraphUrl = makeOldUiGraphUrl(projectId, parsedSelectors);

        const selectorsForSubAlert = parsedSelectors.format();

        let selectorsEl = null;
        if (oldUiGraphUrl) {
          selectorsEl = (<a href={oldUiGraphUrl}>{selectorsForSubAlert}</a>);
        } else {
          selectorsEl = (
            <div>
              <p className="text-danger">{selectorsForSubAlert}</p>
              <div className="alert alert-warning">
                For now, Solomon cannot plot graph for selectors.
              </div>
            </div>
          );
        }

        genericOptionsContent = (
          <div>
            <LabeledValue label="Selectors" value={selectorsEl} />
            {predicateContents}
          </div>
        );
      } catch (e) {
        console.error('something wrong', e);
      }
    } else if (parent.type.expression) {
      const { program = '', checkExpression = '' } = parent.type.expression;

      const subAlertProgram = (
        <SubAlertProgram
          projectId={projectId}
          labels={subAlertLabels}
          code={program}
        />
      );

      const subSubAlertCheckExpression = (
        <SubAlertProgram
          projectId={projectId}
          labels={subAlertLabels}
          code={checkExpression}
        />
      );

      genericOptionsContent = (
        <div>
          {program ? <LabeledValue label="Program" value={subAlertProgram} /> : null}
          {checkExpression ? <LabeledValue label="Check expression" value={subSubAlertCheckExpression} /> : null}
        </div>
      );
    }

    let genericOptionsPanel = null;

    if (genericOptionsContent) {
      genericOptionsPanel = (
        <Panel title="Generic options">
          <DescriptionBlock alert={parent} />
          <DurationDelayBlocks alert={parent} />
          {genericOptionsContent}
          <ChannelsBlock alert={parent} />
          <AnnotationsBlock annotations={parent.annotations} label="Annotations" />
          <AnnotationsBlock annotations={parent.serviceProviderAnnotations} label="Service provider annotations" />
        </Panel>
      );
    }

    return (
      <div className="form-horizontal">
        <Breadcrumb match={this.props.match} alertName={parent.name} />
        <PageTitle title={`Sub alert for ${labelValues}`} />
        <div className="row">
          <div className="col-sm-12">
            <SingleAlertStatus
              projectId={projectId}
              parentId={alertId}
              alertId={subAlertId}
              explanation={subAlertExplanation}
              loadExplanation={this.handleLoadExplanation}
            />
          </div>
        </div>
        <div className="row">
          <div className="col-lg-8">
            {genericOptionsPanel}
            <div className="pull-right">
              <Link
                to={`/admin/projects/${projectId}/mutes/new?alertId=${parent.id}&selectors=${subAlertSelector}`}
                className="btn btn-warning btn-outline"
              >
                Mute
              </Link>
            </div>
          </div>
          <div className="col-lg-4">
            <AlertEvalInfo
              projectId={projectId}
              alertId={alertId}
              subAlertId={subAlertId}
              evalState={evalState}
            />
            <AlertNotificationInfo
              projectId={projectId}
              notificationState={notificationState}
            />
            <AlertMutesPanel evalState={evalState} projectId={projectId} />
          </div>
        </div>
      </div>
    );
  }
}

SubAlertPage.propTypes = {
  match: PropTypes.object.isRequired,
  subAlert: PropTypes.object.isRequired,
  evalState: PropTypes.object.isRequired,
  notificationState: PropTypes.object.isRequired,
  subAlertExplanation: PropTypes.object.isRequired,
  loadSubAlert: PropTypes.func.isRequired,
  loadSubAlertEvalState: PropTypes.func.isRequired,
  loadSubAlertNotificationState: PropTypes.func.isRequired,
  loadExistingSubAlertExplanation: PropTypes.func.isRequired,
  clearSubAlert: PropTypes.func.isRequired,
  clearSubAlertEvalState: PropTypes.func.isRequired,
  clearSubAlertNotificationState: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  subAlert: state.subAlert,
  evalState: state.subAlertEvalState,
  notificationState: state.subAlertNotificationState,
  subAlertExplanation: state.subAlertExplanation,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadSubAlert,
  loadSubAlertEvalState,
  loadSubAlertNotificationState,
  loadExistingSubAlertExplanation,
  clearSubAlert,
  clearSubAlertEvalState,
  clearSubAlertNotificationState,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(SubAlertPage);

export default wrapSubProjectReadPage(connectedPage);
