import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import isEmpty from 'lodash/isEmpty';

import Breadcrumb from '../../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../../components/PageTitle';
import ButtonGroup from '../../../components/ButtonGroup';
import LoadMore from '../../../components/LoadMore/LoadMore';

import { clearSubAlerts, loadNextSubAlerts, loadSubAlerts } from '../../../store/reducers/alerts/subAlerts';
import { loadAlert, clearAlert } from '../../../store/reducers/alerts/alert';

import { formatSearch, parseSearch } from '../../../utils/url';
import SelectorsControl from '../../metrics/SelectorsControl';

import SubAlertsTable from './SubAlertsTable';
import { NOTIFICATION_STATUSES } from '../constants';

import { wrapSubProjectReadPage } from '../../projects/wrapSubProjectPage';

const FIRST_BATCH_SIZE = 100;
const NEXT_BATCH_SIZE = 50;

class SubAlertsNotificationsPage extends PureComponent {
  static mapSearchToState(search) {
    const params = parseSearch(search);

    const filterByLabels = params.get('filterByLabels');
    const filterByNotificationStatus = params.get('filterByNotificationStatus');

    return {
      filterByLabels: filterByLabels || '',
      filterByNotificationStatus: filterByNotificationStatus ? filterByNotificationStatus.split(',') : [],
    };
  }

  static mapStateToSearch(state) {
    const search = {};

    if (state.filterByLabels) {
      search.filterByLabels = state.filterByLabels;
    }

    if (state.filterByNotificationStatus.length > 0) {
      search.filterByNotificationStatus = state.filterByNotificationStatus.join(',');
    }

    return `?${formatSearch(search)}`;
  }

  static mapStateToParams(state) {
    const params = {};

    if (state.filterByLabels) {
      params.filterByLabels = state.filterByLabels;
    }

    if (state.filterByNotificationStatus.length > 0) {
      params.filterByNotificationStatus = state.filterByNotificationStatus.join(',');
    }

    return params;
  }

  constructor(props) {
    super(props);
    this.state = SubAlertsNotificationsPage.mapSearchToState(props.location.search);
  }

  componentDidMount() {
    const { projectId, alertId } = this.props.match.params;
    this.props.loadAlert(projectId, alertId).then(() => this.loadSubAlerts());
  }

  componentDidUpdate(prevProps) {
    const { params } = this.props.match;
    const prevParams = prevProps.match.params;

    if (params.projectId !== prevParams.projectId
      || params.alertId !== prevParams.alertId) {
      this.props.loadAlert(params.projectId, params.alertId);
    }
  }

  componentWillUnmount() {
    this.props.clearAlert();
    this.props.clearSubAlerts();
  }

  loadSubAlerts = () => {
    const { projectId, alertId } = this.props.match.params;

    const params = {
      ...SubAlertsNotificationsPage.mapStateToParams(this.state),
      pageSize: FIRST_BATCH_SIZE,
    };

    this.props.loadSubAlerts(projectId, alertId, params);
  };

  loadNextSubAlerts = () => {
    const { projectId, alertId } = this.props.match.params;

    const params = {
      ...SubAlertsNotificationsPage.mapStateToParams(this.state),
      pageSize: NEXT_BATCH_SIZE,
      pageToken: this.props.subAlertsPage.nextPageToken,
    };

    this.props.loadNextSubAlerts(projectId, alertId, params);
  };

  changeSelectors = (filterByLabels) => {
    this.setState({ filterByLabels }, () => this.reloadSubAlerts());
  };

  changeFilterByNotificationStatus = (filterByNotificationStatus) => {
    this.setState({ filterByNotificationStatus }, () => this.reloadSubAlerts());
  };

  changeUrl = () => {
    this.props.history.replace(SubAlertsNotificationsPage.mapStateToSearch(this.state));
  };

  reloadSubAlerts = () => {
    this.changeUrl();
    this.loadSubAlerts();
  };

  render() {
    const { alert, subAlertsPage, match } = this.props;

    const { projectId, alertId } = match.params;

    if (isEmpty(alert)) {
      return <div>Loading...</div>;
    }

    const { groupByLabels } = alert;

    let body;

    if (isEmpty(alert.groupByLabels)) {
      body = (
        <div className="alert alert-danger">
          This alert does not support sub alerts
        </div>
      );
    } else if (isEmpty(subAlertsPage)) {
      body = <div>Loading...</div>;
    } else {
      const { items: subAlerts, nextPageToken } = subAlertsPage;

      let loadMore = null;

      if (nextPageToken) {
        loadMore = <LoadMore onLoadMore={this.loadNextSubAlerts} />;
      }

      const subAlertsView = (
        <div>
          <SubAlertsTable
            projectId={projectId}
            alertId={alertId}
            groupByLabels={groupByLabels}
            subAlerts={subAlerts}
          />
          {loadMore}
        </div>
      );

      const statuses = NOTIFICATION_STATUSES.filter((s) => s.value !== 'UNKNOWN');

      body = (
        <div className="subalerts-page-content">
          <div className="row subalerts-settings-panel">
            <div className="pull-left btn-toolbar">
              <div className="btn-group subalerts-settings-panel__selectors">
                <SelectorsControl
                  selectors={this.state.filterByLabels}
                  onChange={this.changeSelectors}
                />
              </div>
            </div>
            <div className="btn-toolbar pull-right">
              <ButtonGroup
                choices={statuses.map((s) => s.value)}
                labels={statuses.map((s) => s.title)}
                values={this.state.filterByNotificationStatus}
                onChange={this.changeFilterByNotificationStatus}
              />
            </div>
          </div>
          <div className="row">
            <div className="subalerts-content">
              {subAlertsView}
            </div>
          </div>
        </div>
      );
    }

    return (
      <div>
        <Breadcrumb match={this.props.match} alertName={alert.name} />
        <PageTitle title={`Sub alerts notifications of ${alert.name}`} />
        {body}
      </div>
    );
  }
}

SubAlertsNotificationsPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  alert: PropTypes.object.isRequired,
  subAlertsPage: PropTypes.object.isRequired,
  loadAlert: PropTypes.func.isRequired,
  loadSubAlerts: PropTypes.func.isRequired,
  loadNextSubAlerts: PropTypes.func.isRequired,
  clearAlert: PropTypes.func.isRequired,
  clearSubAlerts: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  alert: state.alert,
  subAlertsPage: state.subAlerts,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadAlert,
  loadSubAlerts,
  loadNextSubAlerts,
  clearAlert,
  clearSubAlerts,
}, dispatch);

export default wrapSubProjectReadPage(
  connect(mapStateToProps, mapDispatchToProps)(SubAlertsNotificationsPage),
);
