import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import isEmpty from 'lodash/isEmpty';
import { Link } from 'react-router-dom';

import Breadcrumb from '../../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../../components/PageTitle';

import { formatSearch, parseSearch } from '../../../utils/url';
import SubAlertsTable from './SubAlertsTable/SubAlertsTable';
import SelectorsControl from '../../metrics/SelectorsControl';
import ButtonGroup from '../../../components/ButtonGroup';
import RadioButtons from '../../../components/RadioButtons';
import LoadMore from '../../../components/LoadMore/LoadMore';
import AnnotationSuggest from '../components/AnnotationSuggest/AnnotationSuggest';
import FullscreenFlexableTiles from '../../../components/Tiles/FullscreenFlexableTiles';
import { roundNumericString } from '../../../utils/number';

import { clearSubAlerts, loadNextSubAlerts, loadSubAlerts } from '../../../store/reducers/alerts/subAlerts';
import { loadAlert, clearAlert } from '../../../store/reducers/alerts/alert';

import './style.css';

const FIRST_BATCH_SIZE = 100;
const NEXT_BATCH_SIZE = 50;
const AUTOREFRESH_DELAY_MILLIS = 30000;

class SubAlertsPage extends PureComponent {
  static mapSearchToState(search) {
    const params = parseSearch(search);

    const view = params.get('view');
    const filterByEvalStatus = params.get('filterByEvalStatus');
    const filterByLabels = params.get('filterByLabels');
    const annotationKeys = params.get('annotationKeys');

    return {
      view: view || 'table',
      filterByEvalStatus: filterByEvalStatus ? filterByEvalStatus.split(',') : [],
      annotationKeys: annotationKeys ? annotationKeys.split(',') : [],
      filterByLabels: filterByLabels || '',
    };
  }

  static mapStateToSearch(state) {
    const search = {};

    search.view = state.view;

    if (!isEmpty(state.filterByEvalStatus)) {
      search.filterByEvalStatus = state.filterByEvalStatus.join(',');
    }

    if (!isEmpty(state.annotationKeys)) {
      search.annotationKeys = state.annotationKeys.join(',');
    }

    if (state.filterByLabels) {
      search.filterByLabels = state.filterByLabels;
    }

    return `?${formatSearch(search)}`;
  }

  static mapStateToParams(state) {
    const params = {};

    if (!isEmpty(state.filterByEvalStatus)) {
      params.filterByEvaluationStatus = state.filterByEvalStatus.join(',');
    }

    if (!isEmpty(state.annotationKeys)) {
      params.annotationKeys = state.annotationKeys.join(',');
    }

    if (state.filterByLabels) {
      params.filterByLabels = state.filterByLabels;
    }

    return params;
  }

  constructor(props) {
    super(props);
    this.state = SubAlertsPage.mapSearchToState(props.location.search);
  }

  componentDidMount() {
    const { projectId, alertId } = this.props.match.params;
    this.props.loadAlert(projectId, alertId).then(() => this.runSubAlertsAutorefresh());
  }

  componentDidUpdate(prevProps) {
    const { params } = this.props.match;
    const prevParams = prevProps.match.params;

    if (params.projectId !== prevParams.projectId
      || params.alertId !== prevParams.alertId) {
      this.clearSubAlertsAutorefresh();
      this.props.loadAlert(params.projectId, params.alertId)
        .then(() => this.runSubAlertsAutorefresh());
    }
  }

  componentWillUnmount() {
    this.clearSubAlertsAutorefresh();
    this.props.clearAlert();
    this.props.clearSubAlerts();
  }

  loadSubAlerts = () => {
    const { projectId, alertId } = this.props.match.params;

    const subAlertsCount = this.props.subAlertsPage.items
      ? this.props.subAlertsPage.items.length
      : 0;
    const pageSize = Math.max(FIRST_BATCH_SIZE, subAlertsCount);

    const params = {
      ...SubAlertsPage.mapStateToParams(this.state),
      pageSize,
    };

    this.props.loadSubAlerts(projectId, alertId, params);
  };

  runSubAlertsAutorefresh = () => {
    try {
      this.loadSubAlerts();
    } catch (e) {
      console.error('failed to refresh sub alerts', e);
    }
    this._autorefreshHandler = window.setTimeout(() => {
      this.runSubAlertsAutorefresh();
    }, AUTOREFRESH_DELAY_MILLIS);
  };

  clearSubAlertsAutorefresh = () => {
    if (this._autorefreshHandler != null) {
      window.clearTimeout(this._autorefreshHandler);
    }
  };

  loadNextSubAlerts = () => {
    const { projectId, alertId } = this.props.match.params;

    const params = {
      ...SubAlertsPage.mapStateToParams(this.state),
      pageSize: NEXT_BATCH_SIZE,
      pageToken: this.props.subAlertsPage.nextPageToken,
    };

    this.props.loadNextSubAlerts(projectId, alertId, params);
  };

  changeView = (view) => {
    this.setState({ view }, () => this.changeUrl());
  };

  changeEvalStatuses = (filterByEvalStatus) => {
    this.setState({ filterByEvalStatus }, () => this.reloadSubAlerts());
  };

  changeSelectors = (filterByLabels) => {
    this.setState({ filterByLabels }, () => this.reloadSubAlerts());
  };

  changeAnnotationKeys = (values) => {
    const newAnnotationKeys = values.map((v) => v.value);
    this.setState({ annotationKeys: newAnnotationKeys }, () => this.reloadSubAlerts());
  };

  changeUrl = () => {
    this.props.history.replace(SubAlertsPage.mapStateToSearch(this.state));
  };

  onMute = (selectors) => {
    const { projectId, alertId } = this.props.match.params;
    this.props.history.push(`/admin/projects/${projectId}/mutes/new?alertId=${alertId}&selectors=${encodeURIComponent(selectors)}`);
  }

  reloadSubAlerts = () => {
    this.changeUrl();
    this.loadSubAlerts();
  };

  render() {
    const {
      alert, subAlertsPage, match,
    } = this.props;

    const { projectId, alertId } = match.params;

    const { pathname } = this.props.location;

    const fullscreen = pathname.endsWith('/fullscreen') || pathname.endsWith('/fullscreen/');

    if (isEmpty(alert)) {
      return <div>Loading...</div>;
    }

    const { groupByLabels } = alert;

    let body;

    if (isEmpty(alert.groupByLabels)) {
      body = (
        <div className="alert alert-danger">
          This alert does not support sub alerts
        </div>
      );
    } else if (isEmpty(subAlertsPage)) {
      body = <div>Loading...</div>;
    } else {
      const { items: subAlerts, nextPageToken } = subAlertsPage;

      let loadMore = null;

      if (nextPageToken) {
        loadMore = <LoadMore onLoadMore={this.loadNextSubAlerts} />;
      }

      const { annotationKeys } = this.state;

      let subAlertsView;

      switch (this.state.view) {
        case 'table':
          subAlertsView = (
            <div>
              <SubAlertsTable
                projectId={projectId}
                alertId={alertId}
                groupByLabels={groupByLabels}
                annotationKeys={this.state.annotationKeys}
                subAlerts={subAlerts}
                isTargetBlank={fullscreen}
              />
              {loadMore}
            </div>
          );
          break;
        case 'tiles': {
          // Magic annotation for old trafficlights compatibility
          const filteredAnnotationKeys = annotationKeys.filter((ann) => ann !== 'trafficLight.color');

          const hasTrafficLightColor = annotationKeys.indexOf('trafficLight.color') >= 0;

          const tiles = subAlerts.map((subAlert) => {
            const {
              id, labels, annotations, evaluationStatusCode, muteStatusCode,
            } = subAlert;

            let color;

            const trafficLightColor = annotations['trafficLight.color'] || '';

            if (hasTrafficLightColor && trafficLightColor) {
              color = trafficLightColor.toUpperCase();
            } else {
              color = evaluationStatusCode;
            }

            const url = `/admin/projects/${projectId}/alerts/${alertId}/subAlerts/${id}`;

            const tileLabels = groupByLabels.map((label) => labels[label] || '-');

            const tileValues = filteredAnnotationKeys.map((ann) => roundNumericString(annotations[ann] || '-'));

            return {
              id,
              labels: tileLabels,
              values: tileValues,
              url,
              color,
              mute: muteStatusCode,
            };
          });

          subAlertsView = (
            <FullscreenFlexableTiles
              tiles={tiles}
              fullscreen={fullscreen}
            />
          );
          break;
        }
        default:
          subAlertsView = null;
      }

      if (fullscreen) {
        body = subAlertsView;
      } else {
        const alertAnnotationKeys = Object.keys(alert.annotations || {});

        body = (
          <div className="subalerts-page-content">
            <div className="row subalerts-settings-panel">
              <div className="pull-left subalerts-settings-panel__selectors">
                <SelectorsControl
                  selectors={this.state.filterByLabels}
                  onChange={this.changeSelectors}
                  showButton
                />
                &nbsp;
                <button
                  className="btn btn-warning btn-outline"
                  type="button"
                  onClick={() => this.onMute(this.state.filterByLabels)}
                >
                  Mute
                </button>
              </div>
              <div className="btn-toolbar pull-right">
                <AnnotationSuggest
                  annotations={alertAnnotationKeys}
                  selectedValues={this.state.annotationKeys}
                  onChange={this.changeAnnotationKeys}
                />
                <ButtonGroup
                  choices={['OK', 'WARN', 'ALARM', 'NO_DATA', 'ERROR']}
                  labels={['OK', 'Warning', 'Alarm', 'No data', 'Error']}
                  values={this.state.filterByEvalStatus}
                  onChange={this.changeEvalStatuses}
                />
                <RadioButtons
                  choices={['table', 'tiles']}
                  labels={[
                    <i className="glyphicon glyphicon glyphicon-align-justify" />,
                    <i className="glyphicon glyphicon-th" />,
                  ]}
                  selected={this.state.view}
                  onSelect={this.changeView}
                />
                <div className="btn-group">
                  <Link
                    className="btn btn-default"
                    to={`/admin/projects/${projectId}/alerts/${alertId}/subAlerts/fullscreen${this.props.location.search}`}
                  >
                    <i className="glyphicon glyphicon-fullscreen" />
                  </Link>
                </div>
              </div>
            </div>
            <div className="row">
              <div className="subalerts-content">
                {subAlertsView}
              </div>
            </div>
          </div>
        );
      }
    }

    if (fullscreen) {
      return body;
    }

    return (
      <div>
        <Breadcrumb match={this.props.match} alertName={alert.name} />
        <PageTitle title={`Sub alerts of ${alert.name}`} />
        {body}
      </div>
    );
  }
}

SubAlertsPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  alert: PropTypes.object.isRequired,
  subAlertsPage: PropTypes.object.isRequired,
  loadAlert: PropTypes.func.isRequired,
  loadSubAlerts: PropTypes.func.isRequired,
  loadNextSubAlerts: PropTypes.func.isRequired,
  clearAlert: PropTypes.func.isRequired,
  clearSubAlerts: PropTypes.func.isRequired,
};

const mapStateToSearch = (state) => ({
  alert: state.alert,
  subAlertsPage: state.subAlerts,
});

const mapDispatchToSearch = (dispatch) => bindActionCreators({
  loadAlert,
  loadSubAlerts,
  loadNextSubAlerts,
  clearAlert,
  clearSubAlerts,
}, dispatch);

export default connect(mapStateToSearch, mapDispatchToSearch)(SubAlertsPage);
