import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';

import AlertEvalStatusLabel from '../../components/AlertEvalStatusLabel';
import LocalDate from '../../../../components/LocalDate';
import Linkifier from '../../../../components/Markdown/Linkifier';

class SubAlertsTable extends PureComponent {
  render() {
    const {
      projectId, alertId,
      groupByLabels, annotationKeys, subAlerts,
      isTargetBlank,
    } = this.props;

    if (subAlerts.length === 0) {
      return null;
    }

    const subAlertsTitle = groupByLabels.length === 1 ? groupByLabels[0] : 'Sub alert';

    return (
      <table className="table table-considered table-hover">
        <thead>
          <tr>
            <th>Eval status</th>
            <th>{subAlertsTitle}</th>
            {annotationKeys.map((ann) => (
              <th key={`ann-${ann}`}>
                {ann}
              </th>
            ))}
            <th>Latest eval</th>
          </tr>
        </thead>
        <tbody>
          {subAlerts.map((subAlert) => {
            const labelsMap = subAlert.labels || {};
            const annotations = subAlert.annotations || {};

            let subAlertTitle;

            if (groupByLabels.length === 1) {
              subAlertTitle = labelsMap[groupByLabels[0]] || '-';
            } else {
              subAlertTitle = groupByLabels.map((label) => `${label}=${labelsMap[label] || '-'}`).join(', ');
            }

            const subAlertLink = (
              <Link
                to={`/admin/projects/${projectId}/alerts/${alertId}/subAlerts/${subAlert.id}`}
                target={isTargetBlank ? '_blank' : '_self'}
                rel="noopener noreferrer"
              >
                {subAlertTitle}
              </Link>
            );

            return (
              <tr key={subAlert.id}>
                <td>
                  <AlertEvalStatusLabel statusCode={subAlert.evaluationStatusCode} />
                </td>
                <td>
                  {subAlertLink}
                </td>
                {annotationKeys.map((ann) => (
                  <td key={`annotation-cell-${ann}`}>
                    <Linkifier text={annotations[ann] || '- (empty)'} />
                  </td>
                ))}
                <td>
                  <LocalDate date={subAlert.latestEval} />
                </td>
              </tr>
            );
          })}
        </tbody>
      </table>
    );
  }
}

SubAlertsTable.propTypes = {
  projectId: PropTypes.string.isRequired,
  alertId: PropTypes.string.isRequired,
  groupByLabels: PropTypes.array.isRequired,
  annotationKeys: PropTypes.array.isRequired,
  subAlerts: PropTypes.array.isRequired,
  isTargetBlank: PropTypes.bool,
};

SubAlertsTable.defaultProps = {
  isTargetBlank: false,
};

export default SubAlertsTable;
