import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import isEmpty from 'lodash/isEmpty';

import { formatSearch, parseSearch } from '../../../utils/url';
import SubAlertsTable from './SubAlertsTable/SubAlertsTable';
import FullscreenFlexableTiles from '../../../components/Tiles/FullscreenFlexableTiles';
import { roundNumericString } from '../../../utils/number';
import { fetchAlert, fetchSubAlerts } from '../../../api/alerts';

import './style.css';

const FIRST_BATCH_SIZE = 100;
const AUTOREFRESH_DELAY_MILLIS = 30000;

class SubAlertsWidget extends PureComponent {
  static mapSearchToState(search) {
    const params = parseSearch(search);

    const view = params.get('view');
    const filterByEvalStatus = params.get('filterByEvalStatus');
    const filterByLabels = params.get('filterByLabels');
    const annotationKeys = params.get('annotationKeys');

    return {
      view: view || 'table',
      filterByEvalStatus: filterByEvalStatus ? filterByEvalStatus.split(',') : [],
      annotationKeys: annotationKeys ? annotationKeys.split(',') : [],
      filterByLabels: filterByLabels || '',
    };
  }

  static mapStateToSearch(state) {
    const search = {};

    search.view = state.view;

    if (!isEmpty(state.filterByEvalStatus)) {
      search.filterByEvalStatus = state.filterByEvalStatus.join(',');
    }

    if (!isEmpty(state.annotationKeys)) {
      search.annotationKeys = state.annotationKeys.join(',');
    }

    if (state.filterByLabels) {
      search.filterByLabels = state.filterByLabels;
    }

    return `?${formatSearch(search)}`;
  }

  static mapStateToParams(state) {
    const params = {};

    if (!isEmpty(state.filterByEvalStatus)) {
      params.filterByEvaluationStatus = state.filterByEvalStatus.join(',');
    }

    if (!isEmpty(state.annotationKeys)) {
      params.annotationKeys = state.annotationKeys.join(',');
    }

    if (state.filterByLabels) {
      params.filterByLabels = state.filterByLabels;
    }

    return params;
  }

  constructor(props) {
    super(props);
    this.state = SubAlertsWidget.mapSearchToState(props.search);
  }

  componentDidMount() {
    this.loadAlert().then(() => this.runSubAlertsAutorefresh());
  }

  async componentDidUpdate(prevProps) {
    const { params } = this.props.match;
    const prevParams = prevProps.match.params;

    if (params.projectId !== prevParams.projectId
      || params.alertId !== prevParams.alertId) {
      this.clearSubAlertsAutorefresh();
      this.loadAlert().then(() => this.runSubAlertsAutorefresh());
    }
  }

  componentWillUnmount() {
    this.clearSubAlertsAutorefresh();
  }

  loadAlert = async () => {
    const { projectId, alertId } = this.props.match.params;

    const alert = await fetchAlert(projectId, alertId);
    this.setState({ alert });
  };

  loadSubAlerts = async () => {
    const { projectId, alertId } = this.props.match.params;

    const pageSize = FIRST_BATCH_SIZE;

    const params = {
      ...SubAlertsWidget.mapStateToParams(this.state),
      pageSize,
    };

    const subAlertsPage = await fetchSubAlerts(projectId, alertId, params);
    this.setState({ subAlertsPage });
  };

  runSubAlertsAutorefresh = async () => {
    try {
      await this.loadSubAlerts();
    } catch (e) {
      console.error('failed to refresh sub alerts', e);
    }
    this._autorefreshHandler = window.setTimeout(() => {
      this.runSubAlertsAutorefresh().then(() => {});
    }, AUTOREFRESH_DELAY_MILLIS);
  };

  clearSubAlertsAutorefresh = () => {
    if (this._autorefreshHandler != null) {
      window.clearTimeout(this._autorefreshHandler);
    }
  };

  render() {
    const { element, match } = this.props;

    const { alert, subAlertsPage } = this.state;

    const { projectId, alertId } = match.params;

    if (isEmpty(alert)) {
      return <div style={{ textAlign: 'center' }}>Loading...</div>;
    }

    const { groupByLabels } = alert;

    let body;

    if (isEmpty(alert.groupByLabels)) {
      body = (
        <div className="alert alert-danger">
          This alert does not support sub alerts
        </div>
      );
    } else if (isEmpty(subAlertsPage)) {
      body = <div style={{ textAlign: 'center' }}>Loading...</div>;
    } else {
      const { items: subAlerts } = subAlertsPage;

      const { annotationKeys } = this.state;

      let subAlertsView;

      switch (this.state.view) {
        case 'table':
          subAlertsView = (
            <div>
              <SubAlertsTable
                projectId={projectId}
                alertId={alertId}
                groupByLabels={groupByLabels}
                annotationKeys={this.state.annotationKeys}
                subAlerts={subAlerts}
                isTargetBlank
              />
            </div>
          );
          break;
        case 'tiles': {
          // Magic annotation for old trafficlights compatibility
          const filteredAnnotationKeys = annotationKeys.filter((ann) => ann !== 'trafficLight.color');

          const hasTrafficLightColor = annotationKeys.indexOf('trafficLight.color') >= 0;

          const tiles = subAlerts.map((subAlert) => {
            const {
              id, labels, annotations, evaluationStatusCode, muteStatusCode,
            } = subAlert;

            let color;

            const trafficLightColor = annotations['trafficLight.color'] || '';

            if (hasTrafficLightColor && trafficLightColor) {
              color = trafficLightColor.toUpperCase();
            } else {
              color = evaluationStatusCode;
            }

            const url = `/admin/projects/${projectId}/alerts/${alertId}/subAlerts/${id}`;

            const tileLabels = groupByLabels.map((label) => labels[label] || '-');

            const tileValues = filteredAnnotationKeys.map((ann) => roundNumericString(annotations[ann] || '-'));

            return {
              id,
              labels: tileLabels,
              values: tileValues,
              url,
              color,
              mute: muteStatusCode,
            };
          });

          subAlertsView = (
            <FullscreenFlexableTiles
              tiles={tiles}
              element={element}
              fullscreen
            />
          );
          break;
        }
        default:
          subAlertsView = null;
      }

      body = subAlertsView;
    }

    return body;
  }
}

SubAlertsWidget.propTypes = {
  element: PropTypes.element.isRequired,
  match: PropTypes.object.isRequired,
  search: PropTypes.string.isRequired,
};

export default SubAlertsWidget;
