/* eslint-disable react/no-did-mount-set-state */
import React from 'react';
import PropTypes from 'prop-types';
import isEqual from 'lodash/isEqual';
import * as http from '../../../../api/http';
import HighchartsChart from '../../../../components/HighchartsChart/HighchartsChart';
import charts from './charts';

export default class PseudoChartkitChart extends React.Component {
  constructor(props) {
    super(props);
    this.state = { loading: false, data: null, error: null };
  }

  componentDidMount() {
    this.loadGraphData();
  }

  componentDidUpdate(prevProps) {
    if (prevProps.id !== this.props.id || !isEqual(prevProps.params, this.props.params)) {
      this.loadGraphData();
    }
  }

  loadGraphData() {
    const { id, params } = this.props;

    const chart = charts[id];

    if (!chart || !chart.urls || !chart.js) {
      this.setState({ loading: false, data: null, error: 'Unknown chart' });
      return;
    }

    const {
      computeTimeRange,
      urls: urlsFunc,
      js: jsFunc,
      highchartsConfig,
    } = chart;

    let urlData;
    let timeRange;
    try {
      timeRange = computeTimeRange(params);
      urlData = urlsFunc(params, timeRange);
    } catch (e) {
      console.error('failed to transform url', e);
      this.setState({ loading: false, data: null, error: 'Failed to transform url' });
      return;
    }

    this.setState({ loading: true, error: null });

    http.sendPost(urlData.url, urlData.body).then((loadedData) => {
      let data;
      let options;
      try {
        data = jsFunc(params, timeRange, loadedData);
        const hasData = data.graphs.length > 1;
        options = highchartsConfig(timeRange, hasData);
        options.title = '';
        options.credits = { enabled: false };
        options.chart.height = '170px';
        if (data.categoriesMs) {
          options.series = data.graphs
            .map((graph) => ({
              ...graph,
              data: graph.data.map((v, i) => [data.categoriesMs[i], v]),
            }));
        } else {
          options.series = data.graphs;
        }
        if (!options.series || options.series.length === 0) {
          options.series = [
            {
              name: 'empty',
              data: [],
            },
          ];
        }
      } catch (e) {
        console.error('failed to transform graph data', e);
        this.setState({ loading: false, data: null, error: 'Failed to transform graph data' });
        return;
      }
      this.setState({ loading: false, data: options, error: null });
    });
  }

  render() {
    const { loading, data, error } = this.state;

    if (loading && !data) {
      return (
        <div>Loading...</div>
      );
    }

    if (error) {
      return (
        <div className="alert alert-danger">{error}</div>
      );
    }

    if (data) {
      return (
        <HighchartsChart options={data} />
      );
    }

    return null;
  }
}

PseudoChartkitChart.propTypes = {
  id: PropTypes.string.isRequired,
  params: PropTypes.object.isRequired,
};
