import { computeTimeRangeImpl, prepareLoadedData } from './utils';

const urls = (params, timeRange) => {
  const {
    project,
    cluster,
    projectId,
    channelId,
  } = params;

  let realProject = project;
  let realCluster = cluster;

  if (cluster === 'cloud_preprod_core') {
    realProject = 'solomon_cloud';
    realCluster = 'preprod';
  } else if (cluster === 'cloud_prod_core') {
    realProject = 'solomon_cloud';
    realCluster = 'production';
  }

  const program = `{cluster="${realCluster}", service="alerting_statuses", sensor="channel.notification.status", projectId="${projectId}", channelId="${channelId}", status="*"}`;

  const { fromMillis, toMillis } = timeRange;

  return {
    url: `/api/v2/projects/${realProject}/sensors/data`,
    body: {
      from: fromMillis,
      to: toMillis,
      program,
      downsampling: {
        aggregation: 'LAST',
        maxPoints: 100,
      },
    },
  };
};

function diff(data) {
  if (data.length < 2) {
    return [];
  }

  const result = [];

  for (let i = 0; i < data.length - 1; ++i) {
    const p1 = data[i];
    const p2 = data[i + 1];

    const x = p1[0];

    const y1 = p1[1];
    const y2 = p2[1];

    let resY;

    if (isNaN(y1) || isNaN(y2) || y2 < y1) {
      resY = NaN;
    } else {
      resY = y2 - y1;
    }

    result.push([x, resY]);
  }

  return result;
}

function convertTimeseries(timeseries) {
  const { status } = timeseries.labels;
  const data = diff(timeseries.data);

  let color;
  let name;
  let z;

  switch (status) {
    case 'SUCCESS':
      color = 'green';
      name = 'Success';
      z = 0;
      break;
    case 'ERROR':
      color = 'red';
      name = 'Error';
      z = 1;
      break;
    case 'INVALID_REQUEST':
      color = 'orange';
      name = 'Invalid request';
      z = 2;
      break;
    // It's very strange that channel status can contain it
    case 'ABSENT_NOTIFICATION_CHANNEL':
      color = 'lightblue';
      name = 'Absent channel';
      z = 3;
      break;
    case 'ERROR_ABLE_TO_RETRY':
      color = 'gray';
      name = 'Retry error';
      z = 4;
      break;
    case 'RESOURCE_EXHAUSTED':
      color = 'purple';
      name = 'Resource exhausted';
      z = 5;
      break;
    case 'OBSOLETE':
      color = '';
      name = 'Obsolete';
      z = 6;
      break;
    default:
      color = '';
      name = status;
      z = 7;
  }

  return {
    name, color, z, data,
  };
}

const js = (params, _, loadedData) => {
  const metrics = prepareLoadedData(loadedData, true);

  const graphs = metrics.map((timeseries) => convertTimeseries(timeseries))
    .sort((a, b) => a.z - b.z);

  return { graphs };
};

const highchartsConfig = (timeRange, hasData) => {
  const { fromMillis, toMillis } = timeRange;
  return {
    xAxis: {
      type: 'datetime',
      startOnTick: false,
      endOnTick: false,
      min: fromMillis,
      max: toMillis,
      gridLineWidth: 0.5,
      gridLineColor: '#a9a9a9',
      gridZIndex: 4,
    },
    yAxis: {
      title: '',
    },
    chart: {
      animation: false,
      type: 'column',
    },
    legend: {
      enabled: hasData,
      align: 'right',
      verticalAlign: 'middle',
      layout: 'vertical',
      itemStyle: { color: 'rgba(0, 0, 0, 0.5)', fontWeight: 'normal' },
    },
    plotOptions: {
      column: {
        groupPadding: 0,
        stacking: 'normal',
      },
      series: {
        animation: false,
      },
    },
    tooltip: {
      shared: true,
      animation: false,
    },
    time: {
      useUTC: false,
    },
  };
};

export default {
  id: 'channel-status',
  computeTimeRange: computeTimeRangeImpl,
  urls,
  js,
  highchartsConfig,
};
