import { computeTimeRangeImpl, prepareLoadedData } from './utils';

const urls = (params, timeRange) => {
  const {
    project,
    cluster,
    projectId,
    alertId,
  } = params;

  const program = `{cluster="${cluster}", service="alerting_statuses", sensor="multiAlert.evaluation.status", projectId="${projectId}", alertId="${alertId}", status!="*:MUTED"}`;

  const { fromMillis, toMillis } = timeRange;

  return {
    url: `/api/v2/projects/${project}/sensors/data`,
    body: {
      from: fromMillis,
      to: toMillis,
      program,
      downsampling: {
        aggregation: 'MAX',
        maxPoints: 100,
      },
    },
  };
};

function convertTimeseries(timeseries) {
  const { status } = timeseries.labels;

  let color;
  let name;
  let z;

  switch (status) {
    case 'OK':
      color = '#5cb85c';
      name = 'OK';
      z = 0;
      break;
    case 'ALARM':
      color = '#d9534f';
      name = 'Alarm';
      z = 1;
      break;
    case 'NO_DATA':
      color = '#5bc0de';
      name = 'No data';
      z = 2;
      break;
    case 'WARN':
      color = '#f0ad4e';
      name = 'Warning';
      z = 3;
      break;
    case 'ERROR':
      color = '#777';
      name = 'Error';
      z = 4;
      break;
    default:
      color = '';
      name = status;
      z = 5;
  }

  return {
    name, color, z, data: timeseries.data,
  };
}

const js = (params, _, loadedData) => {
  const metrics = prepareLoadedData(loadedData, true);

  const graphs = metrics.map((timeseries) => convertTimeseries(timeseries))
    .sort((a, b) => a.z - b.z);

  return { graphs };
};

const highchartsConfig = (timeRange, hasData) => {
  const { fromMillis, toMillis } = timeRange;
  return {
    xAxis: {
      type: 'datetime',
      startOnTick: false,
      endOnTick: false,
      min: fromMillis,
      max: toMillis,
      gridLineWidth: 0.5,
      gridLineColor: '#a9a9a9',
      gridZIndex: 4,
    },
    yAxis: {
      title: '',
    },
    chart: {
      animation: false,
      type: 'column',
    },
    legend: {
      enabled: hasData,
      align: 'right',
      verticalAlign: 'middle',
      layout: 'vertical',
      itemStyle: { color: 'rgba(0, 0, 0, 0.5)', fontWeight: 'normal' },
    },
    plotOptions: {
      column: {
        groupPadding: 0,
        stacking: 'normal',
      },
      series: {
        animation: false,
      },
    },
    tooltip: {
      shared: true,
      animation: false,
    },
    time: {
      useUTC: false,
    },
  };
};

export default {
  id: 'multi-alert-status',
  computeTimeRange: computeTimeRangeImpl,
  urls,
  js,
  highchartsConfig,
};
