/* eslint-disable no-param-reassign */
import { parseDuration } from '../../../../utils/duration';
import { computeRangeFromParamsForOldUi } from '../../../data/utils';
import { prepareLoadedData } from './utils';

const computeTimeRange = (params) => {
  const { time, range } = params;
  const rangeMillis = parseDuration(range, true);

  let b;
  let e;

  if (time) {
    b = time;
    e = new Date(new Date(time).getTime() + rangeMillis).toISOString();
  } else {
    b = range;
    e = '';
  }

  return computeRangeFromParamsForOldUi(b, e, Date.now());
};

const urls = (params, timeRange) => {
  const {
    project,
    cluster,
    projectId,
    alertId,
  } = params;

  const { fromMillis, toMillis } = timeRange;
  const ALERT_STATUS_DIVISOR = 1 << 6; // eslint-disable-line no-bitwise

  const program = `mod({cluster="${cluster}", service="alerting_statuses", sensor="alert.evaluation.status", projectId="${projectId}", alertId="${alertId}"}, ${ALERT_STATUS_DIVISOR})`;

  return {
    url: `/api/v2/projects/${project}/sensors/data`,
    body: {
      from: fromMillis,
      to: toMillis,
      program,
      downsampling: {
        disabled: true,
      },
    },
  };
};

function groupByPeriods(graph, fromMillis, toMillis, gridMillis, timezoneOffset) {
  fromMillis = (fromMillis + timezoneOffset) - (fromMillis % gridMillis);
  toMillis = (toMillis + timezoneOffset) - (toMillis % gridMillis);

  const tss = graph.categoriesMs;
  const { graphs } = graph;

  const newTss = [];
  const newGraphs = graphs.map((g) => ({ ...g, data: [] }));

  let i = 0;

  // eslint-disable-next-line max-len
  for (let curMillis = fromMillis; i < tss.length && curMillis < toMillis; curMillis += gridMillis) {
    while (i < tss.length && tss[i] < curMillis) {
      // eslint-disable-next-line no-plusplus
      i++;
    }

    const bucketValues = new Array(graphs.length);

    for (let j = 0; j < graphs.length; ++j) {
      bucketValues[j] = [];
    }

    const bucketEndMillis = curMillis + gridMillis;

    while (i < tss.length && tss[i] <= bucketEndMillis) {
      for (let j = 0; j < graphs.length; ++j) {
        bucketValues[j].push(graphs[j].data[i]);
      }
      // eslint-disable-next-line no-plusplus
      i++;
    }

    const buckets = bucketValues.map((values) => values.reduce((a, b) => a + b, 0));

    for (let j = 0; j < buckets.length; ++j) {
      newGraphs[j].data.push(buckets[j]);
    }

    newTss.push(curMillis);
  }

  return {
    graphs: newGraphs,
    categoriesMs: newTss,
  };
}

function isEmptyOrZero(data) {
  return data.length === 0 || data.every((v) => v === 0 || isNaN(v));
}

const js = (params, timeRange, loadedData) => {
  const metrics = prepareLoadedData(loadedData, false);

  const { range, timezoneOffset } = params;
  let parsedTimezoneOffset = parseInt(timezoneOffset, 10) * 60 * 1000;

  let gridMillis = 0;

  switch (range) {
    case '1h':
      gridMillis = 30000;
      break;
    case '1d':
      gridMillis = 3600000;
      break;
    case '1w':
      gridMillis = 7 * 3600000;
      break;
    default:
      throw new Error(`unknown range: ${range}`);
  }

  if (range !== '1w') {
    parsedTimezoneOffset = 0;
  }

  const { fromMillis, toMillis } = timeRange;

  let graph;

  if (metrics.length > 0) {
    const timeseries = metrics[0];
    const OK_GRAPH = {
      name: 'OK', z: 1, color: '#5cb85c', data: [],
    };
    const WARN_GRAPH = {
      name: 'Warning', z: 2, color: '#f0ad4e', data: [],
    };
    const ALARM_GRAPH = {
      name: 'Alarm', z: 4, color: '#d9534f', data: [],
    };
    const NO_DATA_GRAPH = {
      name: 'No data', z: 0, color: '#5bc0de', data: [],
    };
    const ERROR_GRAPH = {
      name: 'Error', z: 3, color: '#777', data: [],
    };

    let graphs = [
      OK_GRAPH,
      WARN_GRAPH,
      ALARM_GRAPH,
      NO_DATA_GRAPH,
      ERROR_GRAPH,
    ];

    const graphByKey = {
      0: NO_DATA_GRAPH,
      1: OK_GRAPH,
      3: ERROR_GRAPH,
      4: ALARM_GRAPH,
      6: WARN_GRAPH,
    };

    const categoriesMs = timeseries.data.map((p) => p[0]);
    const values = timeseries.data.map((p) => p[1]);

    for (let i = 0; i < values.length; ++i) {
      const value = values[i];

      const foundGraph = graphByKey[value];

      graphs.forEach((g) => {
        if (foundGraph === g) {
          g.data.push(1);
        } else {
          g.data.push(0);
        }
      });
    }

    graphs = graphs.filter((g) => !isEmptyOrZero(g.data));

    graph = { graphs, categoriesMs };

    if (gridMillis) {
      graph = groupByPeriods(graph, fromMillis, toMillis, gridMillis, parsedTimezoneOffset);
    }
  } else {
    graph = { graphs: [], categoriesMs: [] };
  }

  return graph;
};

const highchartsConfig = (timeRange, hasData) => {
  const { fromMillis, toMillis } = timeRange;

  return {
    chart: {
      type: 'column',
      startOnTick: false,
      endOnTick: false,
      animation: false,
      events: {
        click(event) {
          const ts = event.xAxis[0].value;
          if (!this.synthetic) {
            // eslint-disable-next-line no-param-reassign
            event.pointData = { ts };
          }
        },
      },
    },
    xAxis: {
      type: 'datetime',
      startOnTick: false,
      endOnTick: false,
      min: fromMillis,
      max: toMillis,
      gridLineWidth: 0.5,
      gridLineColor: '#a9a9a9',
      gridZIndex: 4,
    },
    yAxis: {
      visible: false,
    },
    legend: {
      enabled: hasData,
      animation: false,
      align: 'right',
      verticalAlign: 'middle',
      layout: 'vertical',
      itemStyle: { color: 'rgba(0, 0, 0, 0.5)', fontWeight: 'normal' },
    },
    plotOptions: {
      column: {
        groupPadding: 0,
        stacking: 'normal',
      },
      series: {
        animation: false,
        events: {
          click(event) {
            const ts = event.point.x;
            if (!this.synthetic) {
              // eslint-disable-next-line no-param-reassign
              event.pointData = { ts };
            }
          },
        },
      },
    },
    tooltip: {
      shared: true,
      animation: false,
    },
    time: {
      useUTC: false,
    },
    credits: {
      enabled: false,
    },
  };
};

export default {
  id: 'single-alert-status',
  computeTimeRange,
  urls,
  js,
  highchartsConfig,
};
