import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import RadioButtons from '../../../../components/RadioButtons';
import { parseDuration } from '../../../../utils/duration';
import { formatLocalDate } from '../../../../utils/localDate';

function formatTimestampMillis(time) {
  const date = new Date(time);
  const localDateFormatted = formatLocalDate(date);
  return localDateFormatted.replace(' 00:00:00', '');
}

class LimitedTimeBar extends PureComponent {
  handleRangeChange = (range) => {
    this.props.onRangeChange({ range, time: 0 });
  };

  handleBackwardClick = (event) => {
    event.preventDefault();

    const { range, time } = this.props;

    const curTime = time === 0 ? Date.now() : time;
    const nextTime = curTime - parseDuration(range);

    this.props.onRangeChange({ range, time: nextTime });
  };

  handleForwardClick = (event) => {
    event.preventDefault();

    const { range, time } = this.props;

    const curTime = time === 0 ? Date.now() : time;
    const nextTime = curTime + parseDuration(range);

    this.props.onRangeChange({ range, time: nextTime });
  };

  render() {
    const { range, time } = this.props;

    const canNavigateBackForward = range !== '1w';

    let info;

    if (time) {
      const fromLocalDate = formatTimestampMillis(time);
      const toLocalDate = formatTimestampMillis(time + parseDuration(range));

      info = `${fromLocalDate} - ${toLocalDate}`;
    } else {
      switch (range) {
        case '1h':
          info = 'Last hour';
          break;
        case '1d':
          info = 'Last day';
          break;
        case '1w':
          info = 'Last week';
          break;
        default:
          throw new Error(`unknown range ${range}`);
      }
    }

    return (
      <div className="btn-toolbar">
        <div className="btn-group" style={{ marginTop: '5px', marginRight: '10px' }}>
          {info}
        </div>
        {canNavigateBackForward && (
          <div className="btn-group btn-group-sm">
            <button type="button" className="btn btn-default" onClick={this.handleBackwardClick}>＜＜</button>
            <button type="button" className="btn btn-default" onClick={this.handleForwardClick}>＞＞</button>
          </div>
        )}
        <RadioButtons
          choices={['1h', '1d', '1w']}
          labels={['Hour', 'Day', 'Week']}
          selected={range}
          onSelect={this.handleRangeChange}
          size="sm"
        />
      </div>
    );
  }
}

LimitedTimeBar.propTypes = {
  range: PropTypes.string.isRequired,
  time: PropTypes.number.isRequired,
  onRangeChange: PropTypes.func.isRequired,
};

export default LimitedTimeBar;
