/* eslint-disable jsx-a11y/no-static-element-interactions */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import { makeSolomonUrl } from '../utils';
import { formatISO8601 } from '../../../../utils/iso8601';
import { parseDuration } from '../../../../utils/duration';
import LimitedTimeBar from './LimitedTimeBar';
import LinksModal from '../../components/ChartPanel/LinksModal';
import { formatSearch } from '../../../../utils/url';
import PseudoChartkitChart from '../PseudoChartkitChart/PseudoChartkitChart';
import { projectForEnv, alertingClusterForEnv } from '../../../../utils/env';

const CHART_SOURCE = '/editor/solomon/alerting/single-alert-status-3';
const HOUR_MILLIS = 3600 * 1000;

class SingleAlertStatusChart extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { modalShown: false };
  }

  handleStatusChartClick = (event) => {
    const { pointData } = event.nativeEvent;

    if (pointData) {
      let { ts } = pointData;

      switch (this.props.range) {
        case '1w': {
          const timezoneOffsetMillis = new Date().getTimezoneOffset() * 60 * 1000;
          ts -= ((ts - timezoneOffsetMillis) % (24 * HOUR_MILLIS));
          this.props.onRangeChange({ range: '1d', time: ts });
          break;
        }
        case '1d': {
          ts -= ts % HOUR_MILLIS;
          this.props.onRangeChange({ range: '1h', time: ts });
          break;
        }
        default: {
          ts -= ts % 1000;
          this.props.onTimeClick(ts);
        }
      }
    }
  };

  handleShowModal = () => {
    this.setState({ modalShown: true });
  };

  handleHideModal = () => {
    this.setState({ modalShown: false });
  };

  render() {
    const {
      projectId, parentId, alertId, range, time,
    } = this.props;

    const params = {
      project: projectForEnv,
      cluster: alertingClusterForEnv,
      projectId,
      alertId,
      parentId,
      range,
      time: time === 0 ? '' : formatISO8601(time),
      timezoneOffset: new Date().getTimezoneOffset(),
    };

    let b = null;
    let e = null;

    if (time) {
      const fromDate = new Date(time);
      const toDate = new Date(time + parseDuration(range));

      b = fromDate.toISOString();
      e = toDate.toISOString();
    } else {
      b = range;
      e = '';
    }

    const solomonUrl = makeSolomonUrl({
      project: projectForEnv,
      cluster: alertingClusterForEnv,
      service: 'alerting_statuses',
      'l.sensor': 'alert.evaluation.status',
      'l.projectId': projectId,
      'l.parentId': parentId,
      'l.alertId': alertId,
      graph: 'auto',
      stack: 'false',
      downsamplingAggr: 'last',
      b,
      e,
    });

    const chartsUrl = `https://charts.yandex-team.ru/preview${CHART_SOURCE}?${formatSearch(params)}`;

    return (
      <div className="panel panel-default">
        <div className="panel-heading">
          <a href={solomonUrl}>Alert evaluation status</a>
          &nbsp;
          <i role="button" tabIndex={-1} className="glyphicon glyphicon-link" onClick={this.handleShowModal} />
          <div className="pull-right" style={{ marginTop: '-5px' }}>
            <LimitedTimeBar
              range={range}
              time={time}
              onRangeChange={this.props.onRangeChange}
            />
          </div>
        </div>
        <div style={{ height: '170px' }} onClick={this.handleStatusChartClick}>
          <PseudoChartkitChart
            id="single-alert-status"
            params={params}
          />
        </div>
        <LinksModal
          solomonUrl={solomonUrl}
          chartsUrl={chartsUrl}
          isOpen={this.state.modalShown}
          onCancel={this.handleHideModal}
        />
      </div>
    );
  }
}

SingleAlertStatusChart.propTypes = {
  projectId: PropTypes.string.isRequired,
  alertId: PropTypes.string.isRequired,
  parentId: PropTypes.string,
  range: PropTypes.string.isRequired,
  time: PropTypes.number.isRequired,
  onRangeChange: PropTypes.func.isRequired,
  onTimeClick: PropTypes.func.isRequired,
};

SingleAlertStatusChart.defaultProps = {
  parentId: '-',
};

export default SingleAlertStatusChart;
