import React from 'react';
import PropTypes from 'prop-types';
import isEmpty from 'lodash/isEmpty';
import LineChart from './LineChart';
import { fixSeries, LINE_NAMING_MODE } from '../../../../utils/data';
import NoDataExplanation from './NoDataExplanation';
import { ReadonlyTable } from '../AnnotationsTable';
import ScalarTableReadonlyRow from '../ScalarsTable';
import {
  convertAnnotationsMapToList,
  convertAnnotationsMapToListWithType,
} from '../../constants';
import AlertEvalStatusLabel from '../AlertEvalStatusLabel';
import { TabPane, Tabs } from '../../../../components/Tabs';
import Linkifier from '../../../../components/Markdown/Linkifier';
import TableViewer from '../../../../components/TableViewer';

const AlertExplanation = ({
  explanation,
  showStatusLabel,
  showScalars,
  memState,
}) => {
  if (isEmpty(explanation)) {
    return null;
  }

  if (explanation.loading) {
    return <div>Loading...</div>;
  }

  if (isEmpty(explanation.data)) {
    return null;
  }

  const { status } = explanation.data;

  const statusEl = showStatusLabel ? <AlertEvalStatusLabel statusCode={status.code} /> : null;

  let chart;

  if (!isEmpty(explanation.data.series)) {
    const result = fixSeries(
      explanation.data.series,
      { mode: LINE_NAMING_MODE.ALERTING_SPECIALLY },
    );
    const series = result.highchartsLines;
    chart = <LineChart series={series} />;
  } else {
    chart = <NoDataExplanation />;
  }

  let description = null;
  let annotations = null;

  const scalars = (showScalars && !isEmpty(status.scalarValues)) ? (
    <TableViewer
      columns={['Name', 'Value', 'Type']}
      values={status.scalarValues}
      row={ScalarTableReadonlyRow}
      limit={5}
    />
  ) : null;

  if (!isEmpty(status.annotations) && !isEmpty(status.serviceProviderAnnotations)) {
    annotations = (
      <ReadonlyTable
        annotations={convertAnnotationsMapToListWithType(status.annotations, 'User')
          .concat(convertAnnotationsMapToListWithType(status.serviceProviderAnnotations, 'Service'))}
        withTypes
        limit={10}
        linkify
      />
    );
  } else if (!isEmpty(status.annotations) || !isEmpty(status.serviceProviderAnnotations)) {
    annotations = (
      <ReadonlyTable
        annotations={convertAnnotationsMapToList(status.annotations)
          .concat(convertAnnotationsMapToList(status.serviceProviderAnnotations))}
        limit={10}
        linkify
      />
    );
  }

  if (!isEmpty(status.description)) {
    description = (
      <div>
        <pre className="alert alert-danger pre">
          <Linkifier text={status.description} />
        </pre>
      </div>
    );
  }

  let explanationInfo = null;

  if ((description && annotations)
      || (description && scalars)
      || (annotations && scalars)
      || (description && annotations && scalars)) {
    explanationInfo = (
      <Tabs memState={memState}>
        {description && (
          <TabPane label="Description">
            {description}
          </TabPane>
        )}
        {annotations && (
          <TabPane label="Annotations">
            {annotations}
          </TabPane>
        )}
        {scalars && (
          <TabPane label="Scalars">
            {scalars}
          </TabPane>
        )}
      </Tabs>
    );
  } else if (description) {
    explanationInfo = description;
  } else if (annotations) {
    explanationInfo = annotations;
  } else if (scalars) {
    explanationInfo = scalars;
  } else {
    explanationInfo = null;
  }

  return (
    <div className="alert-explanation">
      {statusEl}
      <div style={{ height: '310px' }}>
        {chart}
      </div>
      {explanationInfo}
    </div>
  );
};

AlertExplanation.propTypes = {
  explanation: PropTypes.object.isRequired,
  showStatusLabel: PropTypes.bool,
  showScalars: PropTypes.bool,
  memState: PropTypes.object,
};

AlertExplanation.defaultProps = {
  showStatusLabel: false,
  showScalars: false,
  memState: {
    selected: 0,
  },
};

export default AlertExplanation;
