import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import isEmpty from 'lodash/isEmpty';
import entries from 'lodash/entries';
import { Link } from 'react-router-dom';
import Popover from 'react-bootstrap/es/Popover';
import { connect } from 'react-redux';

import { bindActionCreators } from 'redux';
import OverlayTrigger from 'react-bootstrap/es/OverlayTrigger';
import Panel from '../../../components/Panel';

import LocalDate from '../../../components/LocalDate';
import Modal from '../../../components/Modal';
import { NOTIFICATION_STATUS_BY_VALUE } from '../constants';
import { getProjectFeatureFlags } from '../../../store/reducers/projects/projectFeatureFlags';

class AlertNotificationInfo extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { showModal: false, statusDescription: '' };
  }

  handleShowModal = (event, statusDescription) => {
    event.preventDefault();
    this.setState({ showModal: true, statusDescription });
  };

  handleHideModal = (event) => {
    event.preventDefault();
    this.setState({ showModal: false });
  };

  render() {
    const { projectId, notificationState } = this.props;

    if (isEmpty(notificationState) || isEmpty(notificationState.stateByChannelId)) {
      return null;
    }

    const notificationRows = entries(notificationState.stateByChannelId)
      .filter((entry) => !isEmpty(entry[1]))
      .map((entry) => {
        const channelId = entry[0];
        const channelState = entry[1];

        const statusCode = channelState.status.code;

        let channelTitle;
        let useDefaults;

        if (statusCode === 'ABSENT_NOTIFICATION_CHANNEL') {
          channelTitle = channelId;
          useDefaults = 'true';
        } else {
          channelTitle = (
            <Link to={`/admin/projects/${projectId}/channels/${channelId}`}>
              {channelId}
            </Link>
          );
          useDefaults = 'false';
        }

        const parsedStatus = NOTIFICATION_STATUS_BY_VALUE[statusCode];

        let statusLabel;

        if (parsedStatus) {
          statusLabel = (
            <span className={`label ${parsedStatus.style}`}>
              {parsedStatus.title}
            </span>
          );
        } else {
          statusLabel = (
            <span className="label label-default">
              {statusCode === 'OBSOLETE' ? 'Obsolete' : statusCode}
            </span>
          );
        }

        let defaultChannels;
        if (this.props.projectFeatureFlags.DEFAULT_CHANNELS) {
          defaultChannels = (
            <span>
              <br />
              Sent to default channels:
              {' '}
              {useDefaults}
            </span>
          );
        } else {
          defaultChannels = null;
        }

        if (statusCode !== 'OBSOLETE') {
          const popover = (
            <Popover id={`notification_info_popover_${channelId}`}>
              Channel id:
              {' '}
              {channelId}
              <br />
              Evaluated at:
              {' '}
              <LocalDate date={channelState.latestEval} />
              <br />
              Success at:
              {' '}
              <LocalDate date={channelState.latestSuccess} />
              {defaultChannels}
            </Popover>
          );

          statusLabel = (
            <OverlayTrigger placement="left" overlay={popover}>
              <span className="pointer">
                {statusLabel}
              </span>
            </OverlayTrigger>
          );
        }

        let statusEl = statusLabel;

        if (channelState.status.description) {
          const showDetailsBtn = (
            <span>
              <a
                role="button"
                tabIndex={-1}
                onClick={(event) => this.handleShowModal(event, channelState.status.description)}
              >
                Show details...
              </a>
            </span>
          );

          statusEl = (
            <div>
              {statusLabel}
              <br />
              {showDetailsBtn}
            </div>
          );
        }

        return (
          <tr key={`channel_${channelId}`}>
            <td>{channelTitle}</td>
            <td>{statusEl}</td>
          </tr>
        );
      });

    return (
      <Panel title="Notifications">
        <table className="table table-considered table-hover">
          <thead>
            <tr>
              <th>Channel id</th>
              <th>Status</th>
            </tr>
          </thead>
          <tbody>
            {notificationRows}
          </tbody>
        </table>
        {this.state.showModal && (
          <Modal title="Channel status details" isOpen large onCancel={this.handleHideModal}>
            <pre className="alert alert-danger pre">
              {this.state.statusDescription}
            </pre>
          </Modal>
        )}
      </Panel>
    );
  }
}

AlertNotificationInfo.propTypes = {
  projectId: PropTypes.string.isRequired,
  notificationState: PropTypes.object,
  projectFeatureFlags: PropTypes.object.isRequired,
};

AlertNotificationInfo.defaultProps = {
  notificationState: {},
};

const mapStateToProps = (state) => ({
  projectFeatureFlags: state.projectFeatureFlags,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  getProjectFeatureFlags,
}, dispatch);

export default connect(mapStateToProps, mapDispatchToProps)(AlertNotificationInfo);
