import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import isEmpty from 'lodash/isEmpty';
import { convertAnnotationsMapToList } from '../constants';
import Modal from '../../../components/Modal';
import { ReadonlyTable } from './AnnotationsTable';
import { TabPane, Tabs } from '../../../components/Tabs';
import AlertEvalStatusLabel from './AlertEvalStatusLabel';
import Linkifier from '../../../components/Markdown/Linkifier';

class AlertStatus extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { showModal: false };
  }

  handleShowModalClick = (event) => {
    event.preventDefault();
    this.setState({ showModal: true });
  };

  handleHideModalClick = (event) => {
    event.preventDefault();
    this.setState({ showModal: false });
  };

  render() {
    const { status } = this.props;

    if (isEmpty(status)) {
      return <span>-</span>;
    }

    const canShowDescription = !!status.description;
    const canShowAnnotations = !isEmpty(status.annotations);

    const canShowDetails = canShowDescription || canShowAnnotations;

    let modalContent;

    const descriptionContent = (
      <div>
        <pre className="alert alert-danger pre">
          <Linkifier text={status.description} />
        </pre>
      </div>
    );

    const annotationsContent = (
      <div style={{ maxHeight: '300px', overflow: 'auto' }}>
        <ReadonlyTable
          annotations={convertAnnotationsMapToList(status.annotations)}
          limit={10}
          linkify
        />
      </div>
    );

    if (canShowDescription && canShowAnnotations) {
      modalContent = (
        <Tabs>
          <TabPane label="Description">
            {descriptionContent}
          </TabPane>
          <TabPane label="Annotations">
            {annotationsContent}
          </TabPane>
        </Tabs>
      );
    } else if (canShowDescription) {
      modalContent = descriptionContent;
    } else if (canShowAnnotations) {
      modalContent = annotationsContent;
    } else {
      modalContent = null;
    }

    return (
      <div>
        <p>
          <AlertEvalStatusLabel statusCode={status.code} />
        </p>
        {canShowDetails && (
          <p>
            <a role="button" tabIndex={-1} onClick={this.handleShowModalClick}>
              Show details...
            </a>
          </p>
        )}
        {canShowDetails && (
          <Modal
            title="Details"
            isOpen={this.state.showModal}
            onCancel={this.handleHideModalClick}
            large
          >
            {modalContent}
          </Modal>
        )}
      </div>
    );
  }
}

AlertStatus.propTypes = {
  status: PropTypes.object,
};

AlertStatus.defaultProps = {
  status: {},
};

export default AlertStatus;
