/* eslint-disable max-classes-per-file */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import TableEditor from '../../../components/TableEditor';
import TableViewer from '../../../components/TableViewer';
import Linkifier from '../../../components/Markdown/Linkifier';

const LinkifyContext = React.createContext(false);

const ANNOTATION_SHAPE = PropTypes.shape({
  key: PropTypes.string,
  value: PropTypes.string,
});

const ANNOTATION_TYPED_SHAPE = PropTypes.shape({
  key: PropTypes.string,
  value: PropTypes.string,
  typeName: PropTypes.string,
});

const DEFAULT_ANNOTATION = {
  key: '', value: '',
};

class ReadonlyRow extends PureComponent {
  handleClick = (event) => {
    if (!this.props.onClick) {
      return;
    }
    event.preventDefault();
    event.stopPropagation();
    this.props.onClick(this.props.index);
  };

  handleDelete = (event) => {
    if (!this.props.onDelete) {
      return;
    }
    event.preventDefault();
    event.stopPropagation();
    this.props.onDelete(this.props.index);
  };

  render() {
    const { key, value, typeName } = this.props.value;

    return (
      <tr onClick={this.props.readOnly ? undefined : this.handleClick}>
        <td>
          {key}
        </td>
        <td>
          <div style={{ whiteSpace: 'pre-line', wordBreak: 'break-all' }}>
            <LinkifyContext.Consumer>
              {(linkify) => (linkify ? <Linkifier text={value} /> : value)}
            </LinkifyContext.Consumer>
          </div>
        </td>
        {typeName && (
          <td>
            {typeName}
          </td>
        )}
        {this.props.readOnly ? <td /> : (
          <td className="actions">
            {this.props.onClick && (
              <button type="button" onClick={this.handleClick}>
                <i className="glyphicon glyphicon-pencil" />
              </button>
            )}
            {this.props.onDelete && (
              <button type="button" onClick={this.handleDelete}>
                <i className="glyphicon glyphicon-trash" />
              </button>
            )}
          </td>
        )}
      </tr>
    );
  }
}

ReadonlyRow.propTypes = {
  index: PropTypes.number.isRequired,
  value: ANNOTATION_TYPED_SHAPE.isRequired,
  onClick: PropTypes.func,
  onDelete: PropTypes.func,
  readOnly: PropTypes.bool,
};

ReadonlyRow.defaultProps = {
  onClick: null,
  onDelete: null,
  readOnly: false,
};

class EditableRow extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { ...props.value };
  }

  componentDidMount() {
    this.keyInput.focus();
  }

  onOkClick = (event) => {
    event.preventDefault();
    event.stopPropagation();
    this.doComplete();
  };

  onCancelClick = (event) => {
    event.preventDefault();
    event.stopPropagation();
    this.props.onCancel();
  };

  onKeyDown = (event) => {
    if (event.which === 13) {
      event.preventDefault();
      this.doComplete();
    } else if (event.which === 27) {
      event.preventDefault();
      this.props.onCancel();
    }
  };

  onInputChange = (event) => {
    const { name, value } = event.target;
    this.setState({ [name]: value });
  };

  doComplete() {
    const { onOk, index } = this.props;
    const { key, value } = this.state;
    onOk(index, { key, value });
  }

  render() {
    const { key, value } = this.state;

    return (
      <tr>
        <td colSpan={2}>
          <div className="inside-table-form">
            <div className="form-group">
              <label htmlFor="key">Key</label>
              <input
                type="text"
                name="key"
                className="form-control"
                value={key}
                onChange={this.onInputChange}
                onKeyDown={this.onKeyDown}
                ref={(input) => { this.keyInput = input; }}
              />
            </div>
            <div className="form-group">
              <label htmlFor="value">Value</label>
              <textarea
                name="value"
                className="form-control"
                value={value}
                onChange={this.onInputChange}
                rows={4}
              />
            </div>
          </div>
        </td>
        <td className="actions">
          <button type="button" onClick={this.onOkClick}>
            <i className="glyphicon glyphicon-ok" />
          </button>
          <button type="button" onClick={this.onCancelClick}>
            <i className="glyphicon glyphicon-remove" />
          </button>
        </td>
      </tr>
    );
  }
}

EditableRow.propTypes = {
  index: PropTypes.number.isRequired,
  value: ANNOTATION_SHAPE,
  onOk: PropTypes.func.isRequired,
  onCancel: PropTypes.func.isRequired,
};

EditableRow.defaultProps = {
  value: DEFAULT_ANNOTATION,
};

export const ReadonlyTable = ({
  annotations,
  withTypes,
  linkify,
  limit,
}) => (
  <LinkifyContext.Provider value={linkify}>
    <TableViewer
      columns={withTypes ? ['Key', 'Value', 'Provider', ''] : ['Key', 'Value', '']}
      values={annotations}
      row={ReadonlyRow}
      limit={limit}
    />
  </LinkifyContext.Provider>

);

ReadonlyTable.propTypes = {
  annotations: PropTypes.arrayOf(ANNOTATION_SHAPE),
  withTypes: PropTypes.bool,
  linkify: PropTypes.bool,
  limit: PropTypes.number.isRequired,
};

ReadonlyTable.defaultProps = {
  annotations: [],
  linkify: false,
  withTypes: false,
};

export const EditableTable = ({
  annotations, onDelete, onUpdate, readOnly,
}) => (
  <TableEditor
    columns={['Key', 'Value']}
    values={annotations}
    readonlyRow={ReadonlyRow}
    editableRow={EditableRow}
    responsive={false}
    onDelete={onDelete}
    onUpdate={onUpdate}
    readOnly={readOnly}
  />
);

EditableTable.propTypes = {
  annotations: PropTypes.arrayOf(ANNOTATION_SHAPE),
  onDelete: PropTypes.func.isRequired,
  onUpdate: PropTypes.func.isRequired,
  readOnly: PropTypes.bool,
};

EditableTable.defaultProps = {
  annotations: [],
  readOnly: false,
};

export default {};
