/* eslint-disable max-classes-per-file */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import TableEditor from '../../../components/TableEditor';
import TableViewer from '../../../components/TableViewer';
import FormRadioGroup from '../../graphs/FormRadioGroup';
import FormInput from '../../../components/forms/FormInput';

import {
  getTitleByValue,
  THRESHOLD_TYPE_OPTIONS,
  COMPARISON_OPTIONS,
  TARGET_STATUS_OPTIONS,
  DEFAULT_COMPARISON,
  DEFAULT_THRESHOLD_TYPE,
  DEFAULT_TARGET_STATUS,
} from '../constants';
import AlertEvalStatusLabel from './AlertEvalStatusLabel';

const PREDICATE_RULE_SHAPE = PropTypes.shape({
  thresholdType: PropTypes.string,
  comparison: PropTypes.string,
  threshold: PropTypes.number,
  targetStatus: PropTypes.string,
  thresholdParameterTemplate: PropTypes.string,
});

const DEFAULT_PREDICATE_RULE = {
  thresholdType: DEFAULT_THRESHOLD_TYPE,
  comparison: DEFAULT_COMPARISON,
  threshold: 0,
  targetStatus: DEFAULT_TARGET_STATUS,
};

class ReadonlyRow extends PureComponent {
  handleClick = (event) => {
    if (this.props.readOnly || !this.props.onClick) {
      return;
    }
    event.preventDefault();
    event.stopPropagation();
    this.props.onClick(this.props.index);
  };

  handleDelete = (event) => {
    if (!this.props.onDelete) {
      return;
    }
    event.preventDefault();
    event.stopPropagation();
    this.props.onDelete(this.props.index);
  };

  handleMove = (event, direction) => {
    if (!this.props.onMove) {
      return;
    }
    event.preventDefault();
    event.stopPropagation();
    this.props.onMove(this.props.index, direction);
  };

  handleClone = (event) => {
    if (!this.props.onClone) {
      return;
    }
    event.preventDefault();
    event.stopPropagation();
    this.props.onClone(this.props.index);
  }

  render() {
    const {
      thresholdType, comparison, threshold, targetStatus, thresholdParameterTemplate,
    } = this.props.value;
    const thresholdTypeStr = getTitleByValue(thresholdType, THRESHOLD_TYPE_OPTIONS);
    const comparisonStr = getTitleByValue(comparison, COMPARISON_OPTIONS);

    const actions = this.props.readOnly ? null : (
      <>
        {this.props.onMove && (
          <button type="button" onClick={(ev) => this.handleMove(ev, -1)}>
            <i className="glyphicon glyphicon-arrow-up" />
          </button>
        )}
        {this.props.onMove && (
          <button type="button" onClick={(ev) => this.handleMove(ev, 1)}>
            <i className="glyphicon glyphicon-arrow-down" />
          </button>
        )}
        {this.props.onClone && (
          <button type="button" onClick={this.handleClone}>
            <i className="glyphicon glyphicon-duplicate" />
          </button>
        )}
        {this.props.onDelete && (
          <button type="button" onClick={this.handleDelete}>
            <i className="glyphicon glyphicon-trash" />
          </button>
        )}
      </>
    );

    return (
      <tr onClick={this.handleClick}>
        <td>
          {thresholdTypeStr}
          {' '}
          of points
          {' '}
          {comparisonStr}
          {' '}
          {thresholdParameterTemplate || threshold}
        </td>
        <td>
          <AlertEvalStatusLabel statusCode={targetStatus} />
        </td>
        <td className="actions doublesized">
          {actions}
        </td>
      </tr>
    );
  }
}

ReadonlyRow.propTypes = {
  index: PropTypes.number.isRequired,
  value: PREDICATE_RULE_SHAPE.isRequired,
  onClick: PropTypes.func,
  onDelete: PropTypes.func,
  onMove: PropTypes.func,
  onClone: PropTypes.func,
  readOnly: PropTypes.bool,
};

ReadonlyRow.defaultProps = {
  onClick: null,
  onDelete: null,
  onMove: null,
  onClone: null,
  readOnly: false,
};

class EditableRow extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { ...props.value };
  }

  onOkClick = (event) => {
    event.preventDefault();
    event.stopPropagation();
    this.doComplete();
  };

  onCancelClick = (event) => {
    event.preventDefault();
    event.stopPropagation();
    this.props.onCancel();
  };

  onKeyDown = (event) => {
    if (event.which === 13) {
      event.preventDefault();
      this.doComplete();
    } else if (event.which === 27) {
      event.preventDefault();
      this.props.onCancel();
    }
  };

  onInputChange = (event) => {
    const { name, value } = event.target;
    this.setState({ [name]: value });
  };

  doComplete() {
    const { onOk, index } = this.props;
    const {
      thresholdType, comparison, threshold, targetStatus,
    } = this.state;
    onOk(index, {
      thresholdType, comparison, threshold, targetStatus,
    });
  }

  render() {
    const {
      thresholdType, comparison, threshold, targetStatus,
    } = this.state;

    return (
      <tr>
        <td colSpan={2}>
          <div className="inside-table-form">
            <div className="form-group">
              <FormRadioGroup
                name="thresholdType" label="Aggregation"
                options={THRESHOLD_TYPE_OPTIONS}
                value={thresholdType}
                defaultValue={DEFAULT_THRESHOLD_TYPE}
                onChange={this.onInputChange}
                help="Function that aggregates predicate over timepoints"
              />
              <FormRadioGroup
                name="comparison" label="Comparison"
                options={COMPARISON_OPTIONS}
                value={comparison}
                defaultValue={DEFAULT_COMPARISON}
                onChange={this.onInputChange}
                help="Predicate used to compare a point with the threshold"
              />
              <FormInput
                type="text"
                name="threshold" label="Threshold"
                value={threshold}
                onChange={this.onInputChange}
                help="Target threshold"
              />
              <FormRadioGroup
                name="targetStatus" label="Target status"
                options={TARGET_STATUS_OPTIONS}
                value={targetStatus}
                defaultValue={DEFAULT_TARGET_STATUS}
                onChange={this.onInputChange}
                help="Alert status to set if the predicate is true"
              />
            </div>
          </div>
        </td>
        <td className="actions">
          <button type="button" onClick={this.onOkClick}>
            <i className="glyphicon glyphicon-ok" />
          </button>
          <button type="button" onClick={this.onCancelClick}>
            <i className="glyphicon glyphicon-remove" />
          </button>
        </td>
      </tr>
    );
  }
}

EditableRow.propTypes = {
  index: PropTypes.number.isRequired,
  value: PREDICATE_RULE_SHAPE,
  onOk: PropTypes.func.isRequired,
  onCancel: PropTypes.func.isRequired,
};

EditableRow.defaultProps = {
  value: DEFAULT_PREDICATE_RULE,
};

export const ReadonlyTable = ({ predicateRules, limit }) => (
  <TableViewer
    columns={['Predicate', 'Target status']}
    values={predicateRules}
    row={ReadonlyRow}
    limit={limit}
  />
);

ReadonlyTable.propTypes = {
  predicateRules: PropTypes.arrayOf(PREDICATE_RULE_SHAPE),
  limit: PropTypes.number.isRequired,
};

ReadonlyTable.defaultProps = {
  predicateRules: [],
};

export const EditableTable = ({
  predicateRules, onDelete, onUpdate, onMove, onClone, readOnly,
}) => (
  <TableEditor
    columns={['Predicate', 'Target status']}
    values={predicateRules}
    readonlyRow={ReadonlyRow}
    editableRow={EditableRow}
    onDelete={onDelete}
    onUpdate={onUpdate}
    onMove={onMove}
    onClone={onClone}
    readOnly={readOnly}
  />
);

EditableTable.propTypes = {
  predicateRules: PropTypes.arrayOf(PREDICATE_RULE_SHAPE),
  onDelete: PropTypes.func.isRequired,
  onUpdate: PropTypes.func.isRequired,
  onMove: PropTypes.func,
  onClone: PropTypes.func,
  readOnly: PropTypes.bool,
};

EditableTable.defaultProps = {
  predicateRules: [],
  onMove: null,
  onClone: null,
  readOnly: false,
};

export default {};
