import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import isEmpty from 'lodash/isEmpty';
import { Link } from 'react-router-dom';

import ClickableNotificationStats from '../ClickableNotificationStats';
import NotificationStatusSelector from '../NotificationStatusSelector';
import { ALERT_EVAL_STATUSES, makeJugglerLink } from '../../constants';

const MuteStats = ({ alert }) => ((
  <span>
    {ALERT_EVAL_STATUSES
      .map((status) => {
        const stat = alert.mutedStats[status.statsName] || 0;

        if (!stat) {
          return null;
        }

        return (
          <>
            <span
              key={status.value}
              role="button"
              tabIndex={-1}
              className={`label label-${status.style}`}
            >
              In&nbsp;
              {status.statsName}
              :&nbsp;
              {stat}
            </span>
            &nbsp;
          </>
        );
      })
      .filter(Boolean)}
  </span>
));

MuteStats.propTypes = {
  alert: PropTypes.object.isRequired,
};

const FIRST_BATCH_SIZE = 30;
const NEXT_BATCH_SIZE = 30;

const mapStateToParams = (state) => {
  const params = {};
  if (state.filterByName) {
    params.filterByName = state.filterByName;
  }
  if (state.selectedStatus !== 'ANY') {
    params.filterByNotificationStatus = state.selectedStatus;
  }
  return params;
};

const DEFAULT_STATE = {
  filterByName: '',
  selectedStatus: 'ANY',
};

class RelatedAlertsTable extends PureComponent {
  constructor(props) {
    super(props);
    this.state = DEFAULT_STATE;
  }

  componentDidMount() {
    const params = { pageSize: FIRST_BATCH_SIZE };
    if (this.props.loadAlerts) {
      this.props.loadAlerts(this.props.projectId, this.props.relationId, params);
    }
  }

  componentDidUpdate(prevProps) {
    if (this.props.relationId !== prevProps.relationId) {
      const params = { pageSize: FIRST_BATCH_SIZE };
      this.setState(DEFAULT_STATE, () => {
        this.props.loadAlerts(this.props.projectId, this.props.relationId, params);
      });
    }
  }

  componentWillUnmount() {
    this.props.clearAlerts();
  }

  handleLoadNextAlerts = (event) => {
    event.preventDefault();
    this.loadNextAlerts();
  };

  handleNameFilterChange = (event) => {
    event.preventDefault();
    this.setState({ filterByName: event.target.value }, this.loadAlerts);
  };

  handleStatusFilterChange = (event) => {
    event.preventDefault();
    this.setState({ selectedStatus: event.target.value }, () => {
      this.loadAlerts();
    });
  };

  loadAlerts = () => {
    const { projectId, relationId } = this.props;
    const params = {
      ...mapStateToParams(this.state),
      pageSize: FIRST_BATCH_SIZE,
    };
    this.props.loadAlerts(projectId, relationId, params);
  };

  loadNextAlerts() {
    const { projectId, relationId, alertsPage } = this.props;
    const params = {
      ...mapStateToParams(this.state),
      pageToken: alertsPage.nextPageToken,
      pageSize: NEXT_BATCH_SIZE,
    };
    this.props.loadNextAlerts(projectId, relationId, params);
  }

  render() {
    const {
      projectId,
      alertsPage,
      jugglerParams,
      notificationStats,
      mutedStats,
      warnEmpty,
      emptyMessage,
      filters,
    } = this.props;

    if (isEmpty(alertsPage)) {
      return <span>Loading...</span>;
    }

    const { items, nextPageToken } = alertsPage;

    let loadMoreButton = null;

    if (nextPageToken) {
      loadMoreButton = (
        <button
          type="button"
          className="btn btn-default"
          onClick={this.handleLoadNextAlerts}
        >
          Load more...
        </button>
      );
    }

    const hasJugglerParams = jugglerParams != null;

    if (!items.length && warnEmpty && !this.state.filterByName) {
      return (
        <div className="alert alert-warning" role="alert">
          {emptyMessage}
        </div>
      );
    }

    const alertsTable = (
      <table className="table table-considered table-hover">
        <thead>
          <tr>
            <th>Name</th>
            {notificationStats && <th>Notification stats</th>}
            {mutedStats && <th>Muted stats</th>}
            {hasJugglerParams ? <th>Juggler link</th> : null}
          </tr>
        </thead>
        <tbody>
          {filters && (
            <tr>
              <td>
                <input
                  className="form-control"
                  value={this.state.filterByName}
                  onChange={this.handleNameFilterChange}
                />
              </td>
              {notificationStats && (
                <td>
                  <NotificationStatusSelector
                    value={this.state.selectedStatus}
                    onChange={this.handleStatusFilterChange}
                  />
                </td>
              )}
              {[mutedStats, hasJugglerParams].filter(Boolean).length > 0 && (
                <td colSpan={[mutedStats, hasJugglerParams].filter(Boolean).length} />
              )}
            </tr>
          )}
          {items.map((alert) => (
            <tr key={alert.id}>
              <td>
                <Link to={`/admin/projects/${projectId}/alerts/${alert.id}`}>
                  {alert.name || alert.id}
                </Link>
              </td>
              {notificationStats && (
                <td>
                  <ClickableNotificationStats
                    projectId={alert.projectId}
                    alertId={alert.id}
                    multiAlert={alert.multiAlert}
                    stats={alert.notificationStats}
                  />
                </td>
              )}
              {
                mutedStats && (<td>{alert.multiAlert && <MuteStats alert={alert} />}</td>)
              }
              {hasJugglerParams && (
                <td>
                  {makeJugglerLink({ ...jugglerParams, service: alert.id }, true)}
                </td>
              )}
            </tr>
          ))}
        </tbody>
      </table>
    );

    return (
      <div>
        {alertsTable}
        {loadMoreButton}
      </div>
    );
  }
}

RelatedAlertsTable.propTypes = {
  projectId: PropTypes.string.isRequired,
  relationId: PropTypes.string.isRequired,
  alertsPage: PropTypes.object.isRequired,
  jugglerParams: PropTypes.object,
  loadAlerts: PropTypes.func.isRequired,
  loadNextAlerts: PropTypes.func.isRequired,
  clearAlerts: PropTypes.func.isRequired,
  mutedStats: PropTypes.bool,
  notificationStats: PropTypes.bool,
  warnEmpty: PropTypes.bool,
  filters: PropTypes.bool,
  emptyMessage: PropTypes.string,
};

RelatedAlertsTable.defaultProps = {
  jugglerParams: null,
  mutedStats: false,
  notificationStats: true,
  warnEmpty: false,
  filters: true,
  emptyMessage: 'No alerts found.',
};

export default RelatedAlertsTable;
