import React from 'react';
import isEmpty from 'lodash/isEmpty';
import entries from 'lodash/entries';
import values from 'lodash/values';
import { formatSearch } from '../../utils/url';

export const ONLY_TEMPLATE_FIELDS = [
  'noPointsPolicy',
  'resolvedEmptyPolicy',
  'groupByLabels',
  ['delaySecs', 'delaySeconds'],
  ['windowSecs', 'periodMillis', (x) => x / 1000],
];

export const ALERT_TYPES = {
  THRESHOLD: 'Threshold',
  EXPRESSION: 'Expression',
  FROM_TEMPLATE: 'Preconfigured',
  UNKNOWN: 'Unknown',
};

export const ALERT_TYPE_OPTIONS = [
  { value: ALERT_TYPES.THRESHOLD, upperValue: 'THRESHOLD', title: 'Threshold' },
  { value: ALERT_TYPES.EXPRESSION, upperValue: 'EXPRESSION', title: 'Expression' },
];

export const NO_POINTS_POLICIES = {
  DEFAULT: 'NO_POINTS_DEFAULT',
  OK: 'NO_POINTS_OK',
  WARN: 'NO_POINTS_WARN',
  ALARM: 'NO_POINTS_ALARM',
  NO_DATA: 'NO_POINTS_NO_DATA',
  MANUAL: 'NO_POINTS_MANUAL',
};

export const NO_POINTS_POLICY_OPTIONS = [
  { value: NO_POINTS_POLICIES.DEFAULT, title: 'Default', style: 'default' },
  { value: NO_POINTS_POLICIES.OK, title: 'OK', style: 'success' },
  { value: NO_POINTS_POLICIES.WARN, title: 'Warning', style: 'warning' },
  { value: NO_POINTS_POLICIES.ALARM, title: 'Alarm', style: 'danger' },
  { value: NO_POINTS_POLICIES.NO_DATA, title: 'No data', style: 'info' },
  { value: NO_POINTS_POLICIES.MANUAL, title: 'Manual', style: 'default' },
];

export const RESOLVED_EMPTY_POLICIES = {
  DEFAULT: 'RESOLVED_EMPTY_DEFAULT',
  OK: 'RESOLVED_EMPTY_OK',
  WARN: 'RESOLVED_EMPTY_WARN',
  ALARM: 'RESOLVED_EMPTY_ALARM',
  NO_DATA: 'RESOLVED_EMPTY_NO_DATA',
  MANUAL: 'RESOLVED_EMPTY_MANUAL',
};

export const RESOLVED_EMPTY_POLICY_OPTIONS = [
  { value: RESOLVED_EMPTY_POLICIES.DEFAULT, title: 'Default', style: 'default' },
  { value: RESOLVED_EMPTY_POLICIES.OK, title: 'OK', style: 'success' },
  { value: RESOLVED_EMPTY_POLICIES.WARN, title: 'Warning', style: 'warning' },
  { value: RESOLVED_EMPTY_POLICIES.ALARM, title: 'Alarm', style: 'danger' },
  { value: RESOLVED_EMPTY_POLICIES.NO_DATA, title: 'No data', style: 'info' },
  { value: RESOLVED_EMPTY_POLICIES.MANUAL, title: 'Manual', style: 'default' },
];

export function getAlertType(type = {}, alert = {}) {
  if (type.expression) {
    return ALERT_TYPES.EXPRESSION;
  } if (type.threshold) {
    return ALERT_TYPES.THRESHOLD;
  } if (type.fromTemplate && alert.templateData) {
    return getAlertType(alert.templateData.type);
  }
  return ALERT_TYPES.UNKNOWN;
}

export function parseAlertType(type, alert) {
  return (type === 'FROM_TEMPLATE' && alert.typeData
    ? ALERT_TYPES[alert.typeData.fromTemplate.templateType]
    : ALERT_TYPES[type]) || ALERT_TYPES.UNKNOWN;
}

export function getAlertTypeTitle(alertType, isMultiAlert) {
  if (isMultiAlert) {
    return `Multi ${alertType.toLowerCase()}`;
  }
  return alertType;
}

export const COMPARISON_OPTIONS = [
  { value: 'LT', title: '<' },
  { value: 'LTE', title: '<=' },
  { value: 'EQ', title: '=' },
  { value: 'NE', title: '!=' },
  { value: 'GT', title: '>' },
  { value: 'GTE', title: '>=' },
];

export const DEFAULT_COMPARISON = 'LT';

export const THRESHOLD_TYPE_OPTIONS = [
  { value: 'AT_LEAST_ONE', title: 'At least one' },
  { value: 'AT_ALL_TIMES', title: 'At all times' },
  { value: 'LAST_NON_NAN', title: 'Last' },
  { value: 'AVG', title: 'Avg' },
  { value: 'MIN', title: 'Min' },
  { value: 'MAX', title: 'Max' },
  { value: 'SUM', title: 'Sum' },
  { value: 'COUNT', title: 'Count' },
];

export const DEFAULT_THRESHOLD_TYPE = 'AT_LEAST_ONE';

export const ALERT_STATE_OPTIONS = [
  { value: 'ALL', title: 'All' },
  { value: 'ACTIVE', title: 'Active' },
  { value: 'MUTED', title: 'Muted' },
];

export const getTitleByValue = (value, options) => {
  const filteredOptions = options.filter((o) => o.value === value || o.value === String(value));
  if (filteredOptions.length !== 1) {
    return '';
  }
  return filteredOptions[0].title;
};

export const ALERT_EVAL_STATUS = {
  OK: {
    value: 'OK', title: 'OK', statsName: 'ok', style: 'success',
  },
  WARN: {
    value: 'WARN', title: 'Warning', statsName: 'warn', style: 'warning',
  },
  ALARM: {
    value: 'ALARM', title: 'Alarm', statsName: 'alarm', style: 'danger',
  },
  NO_DATA: {
    value: 'NO_DATA', title: 'No data', statsName: 'noData', style: 'info',
  },
  ERROR: {
    value: 'ERROR', title: 'Error', statsName: 'error', style: 'default',
  },
};

export const ALERT_SEVERITIES = {
  DISASTER: {
    value: 'SEVERITY_DISASTER', title: 'Disaster', style: 'default',
  },
  CRITICAL: {
    value: 'SEVERITY_CRITICAL', title: 'Critical', style: 'default',
  },
  INFO: {
    value: 'SEVERITY_INFO', title: 'Info', style: 'default',
  },
};

export const DEFAULT_TARGET_STATUS = 'ALARM';

export const TARGET_STATUS_OPTIONS = [
  ALERT_EVAL_STATUS.ALARM,
  ALERT_EVAL_STATUS.WARN,
  ALERT_EVAL_STATUS.OK,
  ALERT_EVAL_STATUS.NO_DATA,
];

export const ALERT_EVAL_STATUSES = values(ALERT_EVAL_STATUS);

export const NOTIFICATION_STATUSES = [
  {
    title: 'Success', value: 'SUCCESS', lowerValue: 'success', style: 'label-success',
  },
  {
    title: 'Error', value: 'ERROR', lowerValue: 'error', style: 'label-danger',
  },
  {
    title: 'Invalid request', value: 'INVALID_REQUEST', lowerValue: 'invalidRequest', style: 'label-default',
  },
  {
    title: 'Absent channel', value: 'ABSENT_NOTIFICATION_CHANNEL', lowerValue: 'absentNotificationChannel', style: 'label-default',
  },
  {
    title: 'Retry error', value: 'ERROR_ABLE_TO_RETRY', lowerValue: 'retryError', style: 'label-default',
  },
  {
    title: 'Resource exhausted', value: 'RESOURCE_EXHAUSTED', lowerValue: 'resourceExhausted', style: 'label-default',
  },
];

function makeValueToStatusMap(statuses, valueFunc) {
  const map = {};
  statuses.forEach((status) => {
    map[valueFunc(status)] = status;
  });
  return map;
}

// eslint-disable-next-line max-len
export const NOTIFICATION_STATUS_BY_VALUE = makeValueToStatusMap(NOTIFICATION_STATUSES, (status) => status.value);

export function convertAnnotationsMapToList(map) {
  if (isEmpty(map)) {
    return [];
  }

  return entries(map).map((entry) => ({ key: entry[0], value: entry[1] }));
}

export function convertAnnotationsMapToListWithType(map, type) {
  if (isEmpty(map)) {
    return [];
  }

  return entries(map).map((entry) => ({ key: entry[0], value: entry[1], typeName: type }));
}

export function convertAnnotationsListToMap(list) {
  const map = {};
  list.forEach((item) => { map[item.key] = item.value; });
  return map;
}

export function makeJugglerLink(content, shortMode) {
  const jugglerQueryParts = [];

  jugglerQueryParts.push(`host=${content.host || 'solomon-alert'}`);

  if (content.service) {
    jugglerQueryParts.push(`service=${content.service}`);
  }
  if (content.instance) {
    jugglerQueryParts.push(`instance=${content.instance}`);
  }
  if (content.tags) {
    content.tags.forEach((tag) => {
      jugglerQueryParts.push(`tag=${tag}`);
    });
  }

  const jugglerQuery = jugglerQueryParts.join(' & ');

  let title;

  if (shortMode) {
    title = `service=${content.service || '-'}`;
  } else {
    title = jugglerQuery;
  }

  const href = `https://juggler.yandex-team.ru/raw_events/?${formatSearch({ query: jugglerQuery })}`;

  return (<a href={href}>{title}</a>);
}

export const PARAMETERS_FIELDS = [
  /* Template field, type, input type, field in resulting type */
  ['doubleParameters', 'text', 'number', 'doubleValueParameters'],
  ['intParameters', 'text', 'number', 'intValueParameters'],
  ['textParameters', 'text', 'text', 'textValueParameters'],
  ['labelListParameters', 'list', 'label', 'labelListValueParameters'],
  ['textListParameters', 'list', 'text', 'textListValueParameters'],
];

export const THRESHOLDS_FIELDS = [
  /* Template field, type, input type, field in resulting type */
  ['doubleThresholds', 'text', 'number', 'doubleValueThresholds'],
  ['intThresholds', 'text', 'number', 'intValueThresholds'],
  ['textThresholds', 'text', 'text', 'textValueThresholds'],
  ['labelListThresholds', 'list', 'label', 'labelListValueThresholds'],
  ['textListThresholds', 'list', 'text', 'textListValueThresholds'],
];

export const copyFieldsFromTemplate = (alert, isNew) => {
  const template = alert.templateData;

  if (!template) {
    return alert;
  }

  Object.assign(alert, ONLY_TEMPLATE_FIELDS.reduce((acc, field) => {
    if (Array.isArray(field)) {
      const [name, val, fn = (x) => x] = field;
      acc[name] = fn(template[val]);
    } else {
      acc[field] = template[field];
    }
    return acc;
  }, {}));

  if (isNew) {
    Object.assign(alert, {
      name: template.name,
      description: template.description,
    });
  }

  return alert;
};
