import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import isEmpty from 'lodash/isEmpty';

import {
  clearChannel,
  findChannel,
  saveChannel,
} from '../../store/reducers/channels/channel';
import { parseSearch } from '../../utils/url';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import ChannelForm from './ChannelForm';
import { wrapSubProjectEditPage } from '../projects/wrapSubProjectPage';
import { findTelegramGroupTitles } from '../../store/reducers/alerts/telegramGroupTitles';
import { findYaChatsGroupIds } from '../../store/reducers/alerts/yaChatsGroupIds';

class ChannelEditPage extends PureComponent {
  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
    this._channelId = props.match.params.channelId;
    this._isNew = !this._channelId;

    if (this._isNew) {
      this._channelId = parseSearch(props.location.search).get('duplicateOf');
    }
  }

  componentDidMount() {
    const { channel } = this.props;

    this.props.findTelegramGroupTitles();
    this.props.findYaChatsGroupIds();

    if (this._channelId) {
      const loadedChannelId = channel.id;
      if (loadedChannelId !== this._channelId) {
        this.props.findChannel(this._projectId, this._channelId);
      }
    } else {
      this.props.clearChannel();
    }
  }

  onSubmitChannel = (channel) => {
    this.props.saveChannel(this._projectId, channel, this._isNew)
      .then((payload) => {
        if (payload) {
          this.props.history.push(`/admin/projects/${this._projectId}/channels/${payload.id}`);
        }
      });
  };

  render() {
    const { channel, telegramGroupTitles, yaChatsGroupIds } = this.props;

    if (!this._isNew && isEmpty(channel)) {
      return <div>Loading...</div>;
    }

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={this._isNew ? 'New channel' : `Edit channel ${channel.name}`} />
        <ChannelForm
          projectId={this._projectId}
          channel={channel}
          telegramGroupTitles={telegramGroupTitles}
          yaChatsGroupIds={yaChatsGroupIds}
          isNew={this._isNew}
          onSubmit={this.onSubmitChannel}
        />
      </div>
    );
  }
}

ChannelEditPage.propTypes = {
  match: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  channel: PropTypes.object.isRequired,
  telegramGroupTitles: PropTypes.array.isRequired,
  yaChatsGroupIds: PropTypes.array.isRequired,
  findChannel: PropTypes.func.isRequired,
  findTelegramGroupTitles: PropTypes.func.isRequired,
  findYaChatsGroupIds: PropTypes.func.isRequired,
  saveChannel: PropTypes.func.isRequired,
  clearChannel: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  channel: state.channel,
  telegramGroupTitles: state.telegramGroupTitles,
  yaChatsGroupIds: state.yaChatsGroupIds,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findChannel,
  findTelegramGroupTitles,
  findYaChatsGroupIds,
  saveChannel,
  clearChannel,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(ChannelEditPage);

export default wrapSubProjectEditPage(connectedPage);
