/* eslint-disable max-len */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import { inputTargetValue } from '../../utils/inputs';
import ApiCallExample from '../../components/ApiCallExample';
import FormButtons from '../../components/forms/FormButtons';
import LabelsEditor from '../../components/LabelsEditor';
import FormElement from '../../components/forms/FormElement';
import FormInput from '../../components/forms/FormInput';
import FormRadioGroup from '../graphs/FormRadioGroup';
import FormArea from '../../components/forms/FormArea';
import { EditableTable } from '../../components/NameValueTables/NameValueTables';

import {
  getNotificationMethod,
  METHOD_OPTIONS,
  NOTIFICATION_METHODS,
} from './constants';

import FormDurationInput from '../../components/forms/FormDurationInput';
import UserSuggest from '../../components/UserSuggest';
import FormCheckbox from '../../components/forms/FormCheckbox';
import { isCloud, isProdEnv, isPreEnv } from '../../utils/env';
import FormDatalist from '../../components/forms/FormDatalist';
import Datalist from '../../components/Datalist';
import AlertEvalStatuses from '../alerts/components/AlertEvalStatuses';
import AbcSuggest from '../../components/AbcSuggest/AbcSuggest';
import AbcDutySuggest from '../../components/AbcSuggest/AbcDutySuggest';
import AlertSeverities from '../alerts/components/AlertSeverities';

class ChannelForm extends PureComponent {
  static mapPropsToState(channel, isNew = false) {
    const newChannel = { ...channel };
    delete newChannel.createdAt;
    delete newChannel.updatedAt;
    delete newChannel.state;

    if (isNew) {
      newChannel.id = '';
    }

    if (!newChannel.method) {
      newChannel.method = {};
    }

    const method = getNotificationMethod(newChannel.method) || NOTIFICATION_METHODS.EMAIL;

    let telegramRecipientType;

    if (isNew) {
      telegramRecipientType = 'login';
    } else if (method === NOTIFICATION_METHODS.TELEGRAM) {
      const telegram = newChannel.method.telegram || {};
      telegramRecipientType = telegram.login ? 'login' : 'group';
    }

    let yaChatsRecipientType;

    if (isNew) {
      yaChatsRecipientType = 'login';
    } else if (method === NOTIFICATION_METHODS.YA_CHATS) {
      const yaChats = newChannel.method.yaChats || {};
      yaChatsRecipientType = yaChats.login ? 'login' : 'group';
    }

    let phoneCallRecipientType;

    if (isNew) {
      phoneCallRecipientType = 'login';
    } else if (method === NOTIFICATION_METHODS.PHONE_CALL) {
      const phoneCall = newChannel.method.phoneCall || {};
      phoneCallRecipientType = phoneCall.login ? 'login' : 'duty';
    }

    return {
      channel: newChannel,
      method,
      apiVisibility: true,
      telegramRecipientType,
      yaChatsRecipientType,
      phoneCallRecipientType,
      showDeprecatedFields: false,
    };
  }

  static updateChannelMethod(channel, method, update) {
    const channelMethod = channel.method || {};
    const channelMethodContent = channelMethod[method] || {};
    const newChannelMethodContent = { ...channelMethodContent, ...update };
    const newChannelMethod = { ...channelMethod, [method]: newChannelMethodContent };
    return { ...channel, method: newChannelMethod };
  }

  constructor(props) {
    super(props);
    this.state = ChannelForm.mapPropsToState(props.channel, props.isNew);
    this._nameInputTouched = !!props.channel.name;
  }

  // TODO: make work with state correctly for React 16
  componentDidUpdate(prevProps) {
    if (this.props.projectId !== prevProps.projectId || this.props.channel !== prevProps.channel) {
      this.state = ChannelForm.mapPropsToState(this.props.channel, this.props.isNew);
    }
  }

  onInputChange = (event) => {
    const { target } = event;
    const { name } = target;
    const value = inputTargetValue(target);

    const change = {};
    switch (name) {
      case 'id':
        change.id = value;
        if (!this._nameInputTouched) {
          change.name = value;
        }
        break;
      case 'name':
        this._nameInputTouched = true;
        change.name = value;
        break;
      default:
        change[target.name] = value;
    }

    this.setState({ channel: { ...this.state.channel, ...change } });
  };

  onSwitchDeprecatedFieldsCut = (event) => {
    event.preventDefault();
    this.setState((state) => ({ showDeprecatedFields: !state.showDeprecatedFields }));
  }

  onStatusesChange = (notifyAboutStatuses) => {
    this.setState({ channel: { ...this.state.channel, notifyAboutStatuses } });
  };

  onSeveritiesChange = (defaultForAlertSeverity) => {
    this.setState({ channel: { ...this.state.channel, defaultForAlertSeverity } });
  };

  onRepeatNotifyDelayChange = (repeatNotifyDelayMillis) => {
    this.setState({ channel: { ...this.state.channel, repeatNotifyDelayMillis } });
  };

  onMethodChange = (event) => {
    this.setState({ method: event.target.value });
  };

  onEmailInputChange = (event) => {
    if (event.preventDefault) {
      event.preventDefault();
    }

    const { name } = event.target;
    const value = inputTargetValue(event.target);

    this.setState({ channel: ChannelForm.updateChannelMethod(this.state.channel, 'email', { [name]: value }) });
  };

  onEmailRecipientsChange = (recipients) => {
    this.setState({ channel: ChannelForm.updateChannelMethod(this.state.channel, 'email', { recipients }) });
  };

  onWebhookInputChange = (event) => {
    if (event.preventDefault) {
      event.preventDefault();
    }

    const { name } = event.target;
    const value = inputTargetValue(event.target);

    this.setState({ channel: ChannelForm.updateChannelMethod(this.state.channel, 'webhook', { [name]: value }) });
  };

  onWebhookHeadersChange = (headers) => {
    this.setState({ channel: ChannelForm.updateChannelMethod(this.state.channel, 'webhook', { headers }) });
  };

  onJugglerInputChange = (event) => {
    if (event.preventDefault) {
      event.preventDefault();
    }

    const { name } = event.target;
    const value = inputTargetValue(event.target);

    this.setState({ channel: ChannelForm.updateChannelMethod(this.state.channel, 'juggler', { [name]: value }) });
  };

  onJugglerTagsChange = (tags) => {
    this.setState({ channel: ChannelForm.updateChannelMethod(this.state.channel, 'juggler', { tags }) });
  };

  onSmsInputChange = (event) => {
    event.preventDefault();

    const { name } = event.target;
    const value = inputTargetValue(event.target);

    this.setState({ channel: ChannelForm.updateChannelMethod(this.state.channel, 'sms', { [name]: value }) });
  };

  onSmsLoginChange = (login) => {
    this.setState({ channel: ChannelForm.updateChannelMethod(this.state.channel, 'sms', { login }) });
  };

  onYaChatsInputChange = (event) => {
    event.preventDefault();

    const { name } = event.target;
    const value = inputTargetValue(event.target);

    this.setState({ channel: ChannelForm.updateChannelMethod(this.state.channel, 'yaChats', { [name]: value }) });
  };

  onYaChatsLoginChange = (login) => {
    this.setState({ channel: ChannelForm.updateChannelMethod(this.state.channel, 'yaChats', { login }) });
  };

  onPhoneCallLoginChange = (login) => {
    this.setState({ channel: ChannelForm.updateChannelMethod(this.state.channel, 'phoneCall', { login }) });
  };

  onPhoneCallAbcServiceChange = (abcService) => {
    this.setState({ channel: ChannelForm.updateChannelMethod(this.state.channel, 'phoneCall', { abcService }) });
  };

  onPhoneCallAbcDutySlugChange = (dutySlug) => {
    this.setState({ channel: ChannelForm.updateChannelMethod(this.state.channel, 'phoneCall', { dutySlug }) });
  };

  onYaChatsRecipientTypeChange = (event) => {
    const { value } = event.target;

    const newState = { yaChatsRecipientType: value };

    const hasGroupId = this.state.channel.method
      && this.state.channel.method.yaChats
      && this.state.channel.method.yaChats.groupId;

    if (value === 'group' && !hasGroupId) {
      newState.channel = ChannelForm.updateChannelMethod(this.state.channel, 'yaChats', { groupId: '' });
    }
    this.setState(newState);
  };

  onPhoneCallRecipientTypeChange = (event) => {
    const { value } = event.target;

    const newState = { phoneCallRecipientType: value };
    if (value === 'duty') {
      newState.channel = ChannelForm.updateChannelMethod(this.state.channel, 'phoneCall', { login: '' });
    } else {
      newState.channel = ChannelForm.updateChannelMethod(this.state.channel, 'phoneCall', { abcService: '', dutySlug: '' });
    }
    this.setState(newState);
  };

  onYaChatsGroupIdChange = (event) => {
    const groupId = event.target.value;
    this.setState({ channel: ChannelForm.updateChannelMethod(this.state.channel, 'yaChats', { groupId }) });
  };

  onTelegramInputChange = (event) => {
    const { name } = event.target;
    const value = inputTargetValue(event.target);

    this.setState({ channel: ChannelForm.updateChannelMethod(this.state.channel, 'telegram', { [name]: value }) });
  };

  onTelegramRecipientTypeChange = (event) => {
    const { value } = event.target;

    const newState = { telegramRecipientType: value };

    const hasGroupTitle = this.state.channel.method
      && this.state.channel.method.telegram
      && this.state.channel.method.telegram.groupTitle;

    if (value === 'group' && !hasGroupTitle) {
      newState.channel = ChannelForm.updateChannelMethod(this.state.channel, 'telegram', { groupTitle: '' });
    }
    this.setState(newState);
  };

  onTelegramLoginChange = (login) => {
    this.setState({ channel: ChannelForm.updateChannelMethod(this.state.channel, 'telegram', { login }) });
  };

  onTelegramGroupTitleChange = (event) => {
    const groupTitle = event.target.value;
    this.setState({ channel: ChannelForm.updateChannelMethod(this.state.channel, 'telegram', { groupTitle }) });
  };

  onSubmit = (event) => {
    event.preventDefault();
    this.props.onSubmit(this.getCleanChannel());
  };

  onJsonStateChange = (newChannel) => {
    this.setState(ChannelForm.mapPropsToState(newChannel));
  };

  onChangeVisibility = (visible) => {
    this.setState({ apiVisibility: visible });
  };

  getCleanChannel = () => {
    const { method, channel } = this.state;

    const newChannel = { ...channel };

    const channelMethod = channel.method || {};

    let methodContent;
    switch (method) {
      case NOTIFICATION_METHODS.EMAIL:
        methodContent = { email: channelMethod.email || {} };
        break;
      case NOTIFICATION_METHODS.WEBHOOK:
        methodContent = { webhook: channelMethod.webhook || {} };
        break;
      case NOTIFICATION_METHODS.JUGGLER:
        methodContent = { juggler: channelMethod.juggler || {} };
        break;
      case NOTIFICATION_METHODS.SMS:
        methodContent = { sms: channelMethod.sms || {} };
        break;
      case NOTIFICATION_METHODS.TELEGRAM: {
        const telegram = channelMethod.telegram || {};
        const fixedTelegram = { ...telegram };
        if (this.state.telegramRecipientType === 'login') {
          delete fixedTelegram.groupTitle;
        } else {
          delete fixedTelegram.login;
        }
        methodContent = { telegram: fixedTelegram };
        break;
      }
      case NOTIFICATION_METHODS.YA_CHATS:
        methodContent = { yaChats: channelMethod.yaChats || {} };
        break;
      case NOTIFICATION_METHODS.PHONE_CALL: {
        methodContent = { phoneCall: channelMethod.phoneCall || {} };
        break;
      }
      default:
        methodContent = {};
    }

    newChannel.method = methodContent;

    return newChannel;
  };

  render() {
    const { projectId, isNew } = this.props;

    const { channel, method } = this.state;

    const channelMethod = channel.method || {};

    let methodBlock;

    switch (method) {
      case NOTIFICATION_METHODS.EMAIL: {
        const email = channelMethod.email || {};
        methodBlock = (
          <div>
            <FormElement
              label="Recipients"
              help="List of email's that should receive notifications"
            >
              <LabelsEditor values={email.recipients} onChange={this.onEmailRecipientsChange} />
            </FormElement>
            <FormInput
              type="text"
              name="subjectTemplate" label="Subject"
              value={email.subjectTemplate}
              onChange={this.onEmailInputChange}
              help={(
                <span>
                  Mustache template for subject attribute of email,
                  if not specified will be use global template, e.g.:
                  <code>
                    Send that
                    {' '}
                    <b>&#123;&#123;alert.name&#125;&#125;</b>
                    {' '}
                    is
                    {' '}
                    <b>&#123;&#123;status.code&#125;&#125;</b>
                  </code>
                </span>
)}
            />
            <FormArea
              name="contentTemplate" label="Content"
              value={email.contentTemplate}
              rows={6}
              onChange={this.onEmailInputChange}
              help={(
                <span>
                  Mustache template for email body, if not specified will be use global template,
                  e.g.:
                  <code>
                    Everything is broken:
                    {' '}
                    <b>&#123;&#123;status.code&#125;&#125;</b>
                    !
                  </code>
                </span>
)}
            />
          </div>
        );
        break;
      }
      case NOTIFICATION_METHODS.WEBHOOK: {
        const webhook = channelMethod.webhook || {};
        methodBlock = (
          <div>
            <FormInput
              type="text"
              name="url" label="Url"
              value={webhook.url}
              onChange={this.onWebhookInputChange}
              help="Url that will be use to POST request, e.g.: http://myhost:8181/alert"
            />
            <FormArea
              name="bodyTemplate" label="Body"
              value={webhook.bodyTemplate}
              rows={6}
              onChange={this.onWebhookInputChange}
              help={(
                <span>
                  Mustache template for body, e.g.:
                  <br />
                  <code>
                    &#123;&#34;alertId&#34;: &#34;
                    <b>&#123;&#123;alert.id&#125;&#125;</b>
                    &#34;,
                    &#34;status&#34;: &#34;
                    <b>&#123;&#123;status.code&#125;&#125;</b>
                    &#34;,
                    &#34;when&#34;: &#34;
                    <b>&#123;&#123;since&#125;&#125;</b>
                    &#34;&#125;
                  </code>
                </span>
)}
            />
            <FormElement
              label="Headers"
              help="Additional headers that need inject to notification request"
            >
              <EditableTable
                map={webhook.headers}
                onUpdate={this.onWebhookHeadersChange}
              />
            </FormElement>
          </div>
        );
        break;
      }
      case NOTIFICATION_METHODS.JUGGLER: {
        const juggler = channelMethod.juggler || {};
        methodBlock = (
          <div>
            <FormInput
              type="text"
              name="host" label="Host"
              value={juggler.host}
              onChange={this.onJugglerInputChange}
              help="Jugler host described by mustache template, by default solomon-alert"
            />
            <FormInput
              type="text"
              name="service" label="Service"
              value={juggler.service}
              onChange={this.onJugglerInputChange}
              help="Juggler service described by mustache template, by default {{alert.id}}"
            />
            <FormInput
              type="text"
              name="instance" label="Instance"
              value={juggler.instance}
              onChange={this.onJugglerInputChange}
              help="Juggler instance described by mustache template. Leave empty unless you are absolutely sure you need it"
            />
            <FormElement
              label="Tags"
              help="Mustache template supportable list of juggler tags for event"
            >
              <LabelsEditor values={juggler.tags} onChange={this.onJugglerTagsChange} />
            </FormElement>
            <FormArea
              name="description"
              label="Description"
              value={juggler.description}
              rows={6}
              onChange={this.onJugglerInputChange}
              help="Mustache template, leave empty to use default"
            />
          </div>
        );
        break;
      }
      case NOTIFICATION_METHODS.SMS: {
        const sms = channelMethod.sms || {};

        methodBlock = (
          <div>
            <FormElement
              label="Login"
              help="User login that should receive SMS notifications, will be use default phone from passport"
            >
              <UserSuggest
                value={sms.login}
                onChange={this.onSmsLoginChange}
              />
            </FormElement>
            <FormArea
              name="textTemplate" label="Text template"
              value={sms.textTemplate}
              rows={6}
              onChange={this.onSmsInputChange}
              help={(
                <span>
                  Mustache template for SMS text, e.g.:
                  <code>
                    Everything is broken :
                    {' '}
                    <b>&#123;&#123;status.code&#125;&#125;</b>
                    !
                  </code>
                </span>
)}
            />
          </div>
        );
        break;
      }
      case NOTIFICATION_METHODS.TELEGRAM: {
        const telegram = channelMethod.telegram || {};

        const groupTitleOptions = this.props.telegramGroupTitles;

        let telegramBotName;

        if (isCloud()) {
          telegramBotName = 'SolomonCloudMonitoringBot';
        } else if (isProdEnv()) {
          telegramBotName = 'SolomonMonitoringBot';
        } else if (isPreEnv()) {
          telegramBotName = 'SolomonPreMonitoringBot';
        } else {
          telegramBotName = 'SolomonTestMonitoringBot';
        }

        const telegramBotLink = `http://t.me/${telegramBotName}`;

        const telegramBotAnchor = (
          <a
            href={telegramBotLink}
            rel="noopener noreferrer"
            target="_blank"
          >
            @
            {telegramBotName}
          </a>
        );

        methodBlock = (
          <div>
            <FormElement label="Recipient">
              <label className="radio-inline" htmlFor="telegramLogin">
                <input
                  type="radio"
                  name="telegramRecipientType"
                  id="telegramLogin"
                  value="login"
                  checked={this.state.telegramRecipientType === 'login'}
                  onChange={this.onTelegramRecipientTypeChange}
                />
                {' '}
                Login
              </label>
              <label className="radio-inline" htmlFor="telegramGroup">
                <input
                  type="radio"
                  name="telegramRecipientType"
                  id="telegramGroup"
                  value="group"
                  checked={this.state.telegramRecipientType === 'group'}
                  onChange={this.onTelegramRecipientTypeChange}
                />
                {' '}
                Group title
              </label>
            </FormElement>
            {this.state.telegramRecipientType === 'login' && (
              <>
                <FormElement label="">
                  <span>
                    You should add
                    {' '}
                    {telegramBotAnchor}
                    {' '}
                    to your work telegram account from Staff to receive notifications about alert statuses.
                  </span>
                </FormElement>
                <FormElement label="Login">
                  <UserSuggest
                    value={telegram.login}
                    onChange={this.onTelegramLoginChange}
                  />
                </FormElement>
              </>
            )}
            {this.state.telegramRecipientType === 'group' && (
              <>
                <FormElement label="">
                  <ul>
                    <li>
                      Add
                      {telegramBotAnchor}
                      {' '}
                      to your group
                    </li>
                    <li>Select your group here</li>
                  </ul>
                </FormElement>
                <FormDatalist
                  label="Group title"
                  name="groupTitle"
                  datalist={groupTitleOptions}
                  value={telegram.groupTitle}
                  onChange={this.onTelegramGroupTitleChange}
                />
              </>
            )}
            <FormArea
              name="textTemplate" label="Text template"
              value={telegram.textTemplate}
              rows={6}
              onChange={this.onTelegramInputChange}
              help={(
                <span>
                  Mustache template for Telegram message, e.g.:
                  <code>
                    Everything is broken :
                    {' '}
                    <b>&#123;&#123;status.code&#125;&#125;</b>
                    !
                  </code>
                </span>
)}
            />
            <FormCheckbox
              name="sendScreenshot"
              label="Send screenshot"
              value={telegram.sendScreenshot}
              onChange={this.onTelegramInputChange}
            />
          </div>
        );
        break;
      }
      case NOTIFICATION_METHODS.YA_CHATS: {
        const yaChats = channelMethod.yaChats || {};
        const groupIdOptions = this.props.yaChatsGroupIds.map((group) => ({
          value: group.Id,
          content: (
            <div>
              {group.Title}
              <br />
              <small>{group.Id}</small>
            </div>
          ),
        }));

        methodBlock = (
          <div>
            <FormElement label="Recipient">
              <label className="radio-inline" htmlFor="yaChatsLogin">
                <input
                  type="radio"
                  name="yaChatsRecipientType"
                  id="yaChatsLogin"
                  value="login"
                  checked={this.state.yaChatsRecipientType === 'login'}
                  onChange={this.onYaChatsRecipientTypeChange}
                />
                {' '}
                Login
              </label>
              <label className="radio-inline" htmlFor="yaChatsGroup">
                <input
                  type="radio"
                  name="yaChatsRecipientType"
                  id="yaChatsGroup"
                  value="group"
                  checked={this.state.yaChatsRecipientType === 'group'}
                  onChange={this.onYaChatsRecipientTypeChange}
                />
                {' '}
                Group Id
              </label>
            </FormElement>
            {this.state.yaChatsRecipientType === 'login' && (
              <FormElement
                label="Login"
                help="User login that should receive notifications in Yandex.Messenger"
              >
                <UserSuggest
                  value={yaChats.login}
                  onChange={this.onYaChatsLoginChange}
                />
              </FormElement>
            )}
            {this.state.yaChatsRecipientType === 'group' && (
              <FormElement label="Group Id">
                <Datalist
                  name="groupTitle"
                  datalist={groupIdOptions}
                  value={yaChats.groupId}
                  onChange={this.onYaChatsGroupIdChange}
                />
              </FormElement>
            )}
            <FormArea
              name="textTemplate" label="Text template"
              value={yaChats.textTemplate}
              rows={6}
              onChange={this.onYaChatsInputChange}
              help={(
                <span>
                  Mustache template for notification text, e.g.:
                  <code>
                    Everything is broken :
                    {' '}
                    <b>&#123;&#123;status.code&#125;&#125;</b>
                    !
                  </code>
                </span>
)}
            />
          </div>
        );
        break;
      }
      case NOTIFICATION_METHODS.PHONE_CALL: {
        const phoneCall = channelMethod.phoneCall || {};
        methodBlock = (
          <div>
            <FormElement label="Recipient">
              <label className="radio-inline" htmlFor="phoneCallLogin">
                <input
                  type="radio"
                  name="phoneCallRecipientType"
                  id="phoneCallLogin"
                  value="login"
                  checked={this.state.phoneCallRecipientType === 'login'}
                  onChange={this.onPhoneCallRecipientTypeChange}
                />
                {' '}
                Login
              </label>
              <label className="radio-inline" htmlFor="phoneCallDuty">
                <input
                  type="radio"
                  name="phoneCallRecipientType"
                  id="phoneCallDuty"
                  value="duty"
                  checked={this.state.phoneCallRecipientType === 'duty'}
                  onChange={this.onPhoneCallRecipientTypeChange}
                />
                {' '}
                Duty
              </label>
            </FormElement>
            {this.state.phoneCallRecipientType === 'login' && (
              <FormElement
                label="Login"
                help="User login that should receive notifications by phone call"
              >
                <UserSuggest
                  value={phoneCall.login}
                  onChange={this.onPhoneCallLoginChange}
                />
              </FormElement>
            )}
            {this.state.phoneCallRecipientType === 'duty' && (
            <>
              <FormElement
                label="ABC service"
                help="ABC service slug for duty"
              >
                <AbcSuggest
                  value={phoneCall.abcService}
                  onChange={this.onPhoneCallAbcServiceChange}
                />
              </FormElement>
              <FormElement
                label="ABC duty"
                help="ABC duty slug"
              >
                <AbcDutySuggest
                  value={phoneCall.dutySlug}
                  abcService={phoneCall.abcService}
                  onChange={this.onPhoneCallAbcDutySlugChange}
                />
              </FormElement>
            </>
            )}
          </div>
        );
        break;
      }
      default:
        methodBlock = null;
    }

    return (
      <form className="form-horizontal">
        <div className={this.state.apiVisibility ? 'col-lg-6 col-md-7' : 'col-lg-10 col-md-11'}>
          <div>
            <FormInput
              type="text"
              name="id" label="Id"
              value={channel.id}
              onChange={this.onInputChange}
              help="Unique channel identificator"
              disabled={!isNew}
            />
            <FormInput
              type="text"
              name="name" label="Name"
              value={channel.name} onChange={this.onInputChange}
              help="Human-readable name of notification"
            />
            <FormCheckbox
              name="isDefault"
              label="Default for project"
              value={channel.isDefault}
              onChange={this.onInputChange}
              help={<span>Channel can be used as default for project</span>}
            />
            <FormElement
              label="Default for severities"
              help="Channel will be default for alert severities in suggests"
            >
              <AlertSeverities
                editable
                values={channel.defaultForAlertSeverity || []}
                onChange={this.onSeveritiesChange}
              />
            </FormElement>
            <FormRadioGroup
              name="method"
              label="Method"
              value={method}
              defaultValue={NOTIFICATION_METHODS.EMAIL}
              options={METHOD_OPTIONS}
              onChange={this.onMethodChange}
            />
            {methodBlock}
            <FormElement label="">
              {/* eslint-disable-next-line jsx-a11y/no-static-element-interactions */}
              <a className="pointer" onClick={this.onSwitchDeprecatedFieldsCut}>
                {this.state.showDeprecatedFields ? 'Hide' : 'Show'}
                {' '}
                deprecated fields
              </a>
            </FormElement>
            {this.state.showDeprecatedFields && (
              <>
                <FormDurationInput
                  label="Repeat notify delay"
                  name="repeatNotifyDelayMillis"
                  value={channel.repeatNotifyDelayMillis}
                  onChange={this.onRepeatNotifyDelayChange}
                  help="Delay between repeated notify about the same evaluation status, by default notify about
                  particular evaluation status only after change it. Negative duration means that repeat
                  notification not used"
                />
                <FormElement
                  label="Notify about statuses"
                  help="Notification will send event only about specified alert evaluation statuses"
                >
                  <AlertEvalStatuses
                    editable
                    values={channel.notifyAboutStatuses || []}
                    onChange={this.onStatusesChange}
                  />
                </FormElement>
              </>
            )}
          </div>
          <FormButtons onSubmit={this.onSubmit} />
        </div>
        <div className={this.state.apiVisibility ? 'col-lg-6 col-md-5' : 'col-lg-2 col-md-1'}>
          <ApiCallExample
            path={`/projects/${projectId}/notificationChannels`}
            value={this.getCleanChannel()}
            isNew={this.props.isNew}
            visible={this.state.apiVisibility}
            onChange={this.onJsonStateChange}
            onChangeVisibility={this.onChangeVisibility}
          />
        </div>
      </form>
    );
  }
}

ChannelForm.propTypes = {
  projectId: PropTypes.string.isRequired,
  channel: PropTypes.object.isRequired,
  telegramGroupTitles: PropTypes.array.isRequired,
  yaChatsGroupIds: PropTypes.array.isRequired,
  isNew: PropTypes.bool.isRequired,
  onSubmit: PropTypes.func.isRequired,
};

export default ChannelForm;
