import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import isEmpty from 'lodash/isEmpty';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import LabeledValue from '../../components/LabeledValue';
import Panel from '../../components/Panel';
import LabelsList from '../../components/LabelsList';

import { getNotificationMethod, NOTIFICATION_METHODS } from './constants';

import {
  deleteChannel,
  findChannel,
} from '../../store/reducers/channels/channel';
import {
  clearChannelAlerts,
  loadChannelAlerts,
  loadNextChannelAlerts,
} from '../../store/reducers/channels/channelAlertsPage';

import ShowActions from '../../components/ShowActions';

import ChannelStatusChart from '../alerts/charts/ChannelStatusChart';
import RelatedAlertsTable from '../alerts/components/RelatedAlerts/RelatedAlertsTable';

import { formatDuration } from '../../utils/duration';
import { makeJugglerLink } from '../alerts/constants';
import EntityInfo from '../../components/EntityInfo';
import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';
import { PROJECT_PERMISSIONS } from '../../auth/ProjectPermissions';
import UserName from '../../components/UserName';
import AlertEvalStatuses from '../alerts/components/AlertEvalStatuses';

class ChannelPage extends PureComponent {
  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
    this._channelId = props.match.params.channelId;
  }

  componentDidMount() {
    this.props.findChannel(this._projectId, this._channelId);
  }

  componentDidUpdate(prevProps) {
    if (this.props.match.params.channelId !== prevProps.match.params.channelId) {
      this._projectId = this.props.match.params.projectId;
      this._channelId = this.props.match.params.channelId;
      this.props.findChannel(this._projectId, this._channelId);
    }
  }

  onDeleteChannelClick = (event) => {
    event.preventDefault();
    this.props.deleteChannel(this._projectId, this._channelId)
      .then((action) => {
        if (action) {
          this.props.history.push(`/admin/projects/${this._projectId}/channels`);
        }
      });
  };

  render() {
    const { channel, projectAuth } = this.props;

    if (isEmpty(channel)) {
      return <div>Loading...</div>;
    }

    const canEdit = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_UPDATE);
    const canDelete = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_DELETE);

    const method = getNotificationMethod(channel.method);

    let mainBlock;

    switch (method) {
      case NOTIFICATION_METHODS.EMAIL: {
        const content = channel.method.email;
        mainBlock = (
          <div>
            <LabeledValue label="Recipients" value={<LabelsList values={content.recipients} />} />
            <LabeledValue label="Subject template" value={content.subjectTemplate} />
            <LabeledValue label="Content template" value={content.contentTemplate} />
          </div>
        );
        break;
      }
      case NOTIFICATION_METHODS.JUGGLER: {
        const content = channel.method.juggler;

        const jugglerLink = makeJugglerLink(content);

        mainBlock = (
          <div>
            <LabeledValue label="Juggler query" value={jugglerLink} />
            {content.description && (
              <LabeledValue label="Description" value={content.description} />
            )}
          </div>
        );
        break;
      }
      case NOTIFICATION_METHODS.WEBHOOK: {
        const content = channel.method.webhook;
        mainBlock = (
          <div>
            <LabeledValue label="Url" value={content.url} />
            <LabeledValue label="Body template" value={content.bodyTemplate} />
            <LabeledValue label="Headers" value={JSON.stringify(content.headers)} />
          </div>
        );
        break;
      }
      case NOTIFICATION_METHODS.SMS: {
        const content = channel.method.sms;
        mainBlock = (
          <div>
            {content.phone ? <LabeledValue label="Phone" value={content.phone} /> : null}
            {content.login ? <LabeledValue label="Login" value={<UserName login={content.login} />} /> : null}
            {content.textTemplate ? <LabeledValue label="Text template" value={content.textTemplate} /> : null}
          </div>
        );
        break;
      }
      case NOTIFICATION_METHODS.TELEGRAM: {
        const content = channel.method.telegram;
        mainBlock = (
          <div>
            {content.login ? <LabeledValue label="Login" value={<UserName login={content.login} />} /> : null}
            {content.groupTitle ? <LabeledValue label="Group title" value={content.groupTitle} /> : null}
            {content.textTemplate ? <LabeledValue label="Text template" value={content.textTemplate} /> : null}
          </div>
        );
        break;
      }
      case NOTIFICATION_METHODS.YA_CHATS: {
        const content = channel.method.yaChats;
        mainBlock = (
          <div>
            {content.login ? <LabeledValue label="Login" value={<UserName login={content.login} />} /> : null}
            {content.groupId ? <LabeledValue label="Group ID" value={<a href={`https://q.yandex-team.ru/#chats/${encodeURIComponent(content.groupId)}`}>{content.groupId}</a>} /> : null}
            {content.textTemplate ? <LabeledValue label="Text template" value={content.textTemplate} /> : null}
          </div>
        );
        break;
      }
      case NOTIFICATION_METHODS.PHONE_CALL: {
        const content = channel.method.phoneCall;
        mainBlock = (
          <div>
            {content.login ? <LabeledValue label="Login" value={<UserName login={content.login} />} /> : null}
            {content.abcService ? <LabeledValue label="ABC service duty" value={<a href={`https://abc.yandex-team.ru/services/${encodeURIComponent(content.abcService)}/duty/`}>{content.abcService}</a>} /> : null}
            {content.dutySlug ? <LabeledValue label="Duty slug" value={content.dutySlug} /> : null}
          </div>
        );
        break;
      }
      default:
        mainBlock = null;
    }

    let jugglerParams = null;

    if (method === NOTIFICATION_METHODS.JUGGLER && !channel.method.juggler.service) {
      jugglerParams = channel.method.juggler;
    }

    let statuses = null;

    if (!isEmpty(channel.notifyAboutStatuses)) {
      const statusesContent = <AlertEvalStatuses values={channel.notifyAboutStatuses} />;
      statuses = (
        <LabeledValue label="Notify about statuses" value={statusesContent} />
      );
    }

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Channel ${channel.name}`} />
        <div className="row">
          <div className="col-lg-12">
            <ChannelStatusChart projectId={this._projectId} channelId={this._channelId} />
          </div>
        </div>
        <div className="row">
          <div className="col-lg-6">
            <Panel title="Generic options">
              <LabeledValue label="Id" value={channel.id} />
              {statuses}
              {channel.repeatNotifyDelayMillis > 0 && (
                <LabeledValue label="Repeat notify delay" value={formatDuration(channel.repeatNotifyDelayMillis)} />
              )}
              <LabeledValue label="Method" value={method} />
              {mainBlock}
              {channel.isDefault
                ? <LabeledValue label="Default for project" value="Yes" />
                : null}
              <EntityInfo entity={channel} />
            </Panel>
            <ShowActions
              editPath={`/admin/projects/${this._projectId}/channels/${this._channelId}/edit`}
              duplicatePath={`/admin/projects/${this._projectId}/channels/new?duplicateOf=${this._channelId}`}
              onActivate={() => {}}
              onDeactivate={() => {}}
              onDelete={this.onDeleteChannelClick}
              canEdit={canEdit}
              canChangeState={false}
              canDelete={canDelete}
              isActive
            />
          </div>
          <div className="col-lg-6">
            <Panel title="Related alerts">
              <RelatedAlertsTable
                projectId={this._projectId}
                relationId={this._channelId}
                alertsPage={this.props.channelAlertsPage}
                loadAlerts={this.props.loadChannelAlerts}
                loadNextAlerts={this.props.loadNextChannelAlerts}
                clearAlerts={this.props.clearChannelAlerts}
                jugglerParams={jugglerParams}
              />
            </Panel>
          </div>
        </div>
      </div>
    );
  }
}

ChannelPage.propTypes = {
  match: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  channel: PropTypes.object.isRequired,
  channelAlertsPage: PropTypes.object.isRequired,
  findChannel: PropTypes.func.isRequired,
  deleteChannel: PropTypes.func.isRequired,
  loadChannelAlerts: PropTypes.func.isRequired,
  loadNextChannelAlerts: PropTypes.func.isRequired,
  clearChannelAlerts: PropTypes.func.isRequired,
  projectAuth: PropTypes.object.isRequired,
};

const mapStateToProps = (state) => ({
  channel: state.channel,
  channelAlertsPage: state.channelAlertsPage,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findChannel,
  deleteChannel,
  loadChannelAlerts,
  loadNextChannelAlerts,
  clearChannelAlerts,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(ChannelPage);

export default wrapSubProjectReadPage(connectedPage);
