import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import { Link } from 'react-router-dom';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import { getNotificationMethod, NOTIFICATION_METHODS } from './constants';

import {
  findProjectChannels,
  findNextProjectChannels,
} from '../../store/reducers/channels/channelsPage';

import LabelsList from '../../components/LabelsList';
import { formatSearch, parseSearch } from '../../utils/url';
import ButtonGroup from '../../components/ButtonGroup';
import LoadMore from '../../components/LoadMore/LoadMore';
import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';
import { PROJECT_PERMISSIONS } from '../../auth/ProjectPermissions';

class ChannelsPage extends PureComponent {
  static mapSearchToState = (search) => {
    const parsedSearch = parseSearch(search);
    const text = parsedSearch.get('text') || '';
    const types = parsedSearch.get('types') || '';
    const selectedTypes = types ? types.split(',') : [];
    const pageSize = parsedSearch.get('pageSize') || '30';
    return { text, selectedTypes, pageSize };
  };

  static mapStateToParams = (state) => {
    const params = {};
    if (state.text) {
      params.filterByName = state.text;
    }
    if (state.selectedTypes.length > 0) {
      params.filterByType = state.selectedTypes.join(',');
    }
    params.pageSize = state.pageSize;
    return params;
  };

  static mapStateToSearch = (state) => {
    const search = {};
    if (state.text) {
      search.text = state.text;
    }
    if (state.selectedTypes.length > 0) {
      search.types = state.selectedTypes.join(',');
    }
    if (state.pageSize !== '30') {
      search.pageSize = state.pageSize;
    }
    return search;
  };

  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
    this.state = ChannelsPage.mapSearchToState(props.location.search);
  }

  componentDidMount() {
    this.inputEl.focus();
    this.props.findProjectChannels(this._projectId, ChannelsPage.mapStateToParams(this.state));
  }

  onTextChange = (event) => {
    event.preventDefault();
    const text = event.target.value;
    this.changeState({ text });
  };

  onTypesChange = (values) => {
    this.changeState({ selectedTypes: values });
  };

  onLoadMore = () => {
    const pageToken = this.props.channelsPage.nextPageToken;
    const newParams = {
      ...ChannelsPage.mapStateToParams(this.state),
      pageToken,
    };
    this.props.findNextProjectChannels(this._projectId, newParams);
  };

  changeState = (newStateChange) => {
    this.setState(newStateChange, () => {
      const search = ChannelsPage.mapStateToSearch(this.state);
      const params = ChannelsPage.mapStateToParams(this.state);
      this.props.history.replace(`?${formatSearch(search)}`);
      this.props.findProjectChannels(this._projectId, params);
    });
  };

  shortMethodPreview = (method) => {
    const methodType = getNotificationMethod(method);

    switch (methodType) {
      case NOTIFICATION_METHODS.EMAIL: {
        return (
          <LabelsList values={method.email.recipients} />
        );
      }
      case NOTIFICATION_METHODS.JUGGLER: {
        const { juggler } = method;

        const labels = [];

        if (juggler.host) {
          labels.push(`host=${juggler.host}`);
        }
        if (juggler.service) {
          labels.push(`service=${juggler.service}`);
        }
        if (juggler.tags && juggler.tags.length > 0) {
          labels.push(`tags=${juggler.tags.join('|')}`);
        }

        return <LabelsList values={labels} />;
      }
      case NOTIFICATION_METHODS.WEBHOOK: {
        return (
          <div className="label label-success">
            {method.webhook.url}
          </div>
        );
      }
      case NOTIFICATION_METHODS.SMS: {
        if (method.sms.login) {
          return (
            <a href={`https://staff.yandex-team.ru/${method.sms.login}`}>
              <div className="label label-info">
                {method.sms.login}
              </div>
            </a>
          );
        }

        return (
          <div className="label label-info">
            {method.sms.phone}
          </div>
        );
      }
      case NOTIFICATION_METHODS.TELEGRAM: {
        if (method.telegram.login) {
          return (
            <a href={`https://staff.yandex-team.ru/${method.telegram.login}`}>
              <div className="label label-info">
                {method.telegram.login}
              </div>
            </a>
          );
        }

        return (
          <div className="label label-info">
            {method.telegram.groupTitle}
          </div>
        );
      }
      case NOTIFICATION_METHODS.YA_CHATS: {
        if (method.yaChats.login) {
          return (
            <a href={`https://staff.yandex-team.ru/${method.yaChats.login}`}>
              <div className="label label-info">
                {method.yaChats.login}
              </div>
            </a>
          );
        }

        const chatUrl = `https://q.yandex-team.ru/#chats/${encodeURIComponent(method.yaChats.groupId)}`;
        return (
          <a href={chatUrl}>
            <div className="label label-info">
              {method.yaChats.groupId}
            </div>
          </a>
        );
      }
      case NOTIFICATION_METHODS.PHONE_CALL: {
        if (method.phoneCall.login) {
          return (
            <a href={`https://staff.yandex-team.ru/${method.phoneCall.login}`}>
              <div className="label label-info">
                {method.phoneCall.login}
              </div>
            </a>
          );
        }
        const serviceUrl = `https://abc.yandex-team.ru/services/${encodeURIComponent(method.phoneCall.abcService)}/duty/`;
        const value = `${method.phoneCall.abcService}/${method.phoneCall.dutySlug}`;
        return (
          <a href={serviceUrl}>
            <div className="label label-info">
              {value}
            </div>
          </a>
        );
      }
      default:
        return null;
    }
  };

  render() {
    const { channelsPage, projectAuth } = this.props;
    const channels = channelsPage.items || [];

    const canCreate = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_UPDATE);

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Channels of ${this.props.projectName}`} />

        <div className="btn-toolbar table-toolbar">
          {canCreate && (
            <div className="btn-group">
              <Link to={`/admin/projects/${this._projectId}/channels/new`} className="btn btn-outline btn-success">
                <i className="glyphicon glyphicon-plus" />
                {' '}
                Create New
              </Link>
            </div>
          )}
          <div className="btn-group">
            <input
              className="form-control"
              value={this.state.text}
              onChange={this.onTextChange}
              ref={(inputEl) => { this.inputEl = inputEl; }}
            />
          </div>
          <div className="btn-group pull-right">
            <ButtonGroup
              choices={['EMAIL', 'SMS', 'JUGGLER', 'WEBHOOK', 'TELEGRAM', 'YA_CHATS', 'PHONE_CALL']}
              labels={['E-Mail', 'SMS', 'Juggler', 'Webhook', 'Telegram', 'Yandex.Messenger', 'Phone']}
              values={this.state.selectedTypes}
              onChange={this.onTypesChange}
            />
          </div>
        </div>

        <table className="table table-condensed table-hover">
          <thead>
            <tr>
              <th>Name</th>
              <th>Method</th>
              <th>Preview</th>
            </tr>
          </thead>
          <tbody>
            {channels.map((channel, index) => (
              <tr key={channel.id}>
                <td>
                  <Link to={`/admin/projects/${this._projectId}/channels/${channel.id}`} tabIndex={index + 1}>
                    {channel.name}
                  </Link>
                </td>
                <td>{getNotificationMethod(channel.method)}</td>
                <td>{this.shortMethodPreview(channel.method)}</td>
              </tr>
            ))}
          </tbody>
        </table>
        {channelsPage.nextPageToken
          ? (<LoadMore onLoadMore={this.onLoadMore} />)
          : null}
      </div>
    );
  }
}

ChannelsPage.propTypes = {
  history: PropTypes.object.isRequired,
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  projectName: PropTypes.string.isRequired,
  channelsPage: PropTypes.object.isRequired,
  findProjectChannels: PropTypes.func.isRequired,
  findNextProjectChannels: PropTypes.func.isRequired,
  projectAuth: PropTypes.object.isRequired,
};

const mapStateToProps = (state) => ({
  projectName: state.projectData.project.name,
  channelsPage: state.channelsPage,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findProjectChannels,
  findNextProjectChannels,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(ChannelsPage);

export default wrapSubProjectReadPage(connectedPage);
