import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';
import isEmpty from 'lodash/isEmpty';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import ClusterForm from './ClusterForm';
import { parseSearch } from '../../utils/url';
import { wrapSubProjectEditPage } from '../projects/wrapSubProjectPage';

import { findCluster, saveCluster, clearCluster } from '../../store/reducers/clusters/cluster';
import { resolveCluster, clearResolvedCluster } from '../../store/reducers/clusters/resolvedCluster';

class ClusterEditPage extends PureComponent {
  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
    this._clusterId = props.match.params.clusterId;
    this._isNewCluster = !this._clusterId;

    if (this._isNewCluster) {
      this._clusterId = parseSearch(props.location.search).get('duplicateOf');
    }
  }

  componentDidMount() {
    const { cluster } = this.props;
    if (this._clusterId) {
      const loadedClusterId = cluster.id;
      if (loadedClusterId !== this._clusterId) {
        this.props.findCluster(this._projectId, this._clusterId);
      }
    } else {
      this.props.clearCluster();
    }
  }

  onSubmit = (value) => {
    const { history } = this.props;
    this.props.saveCluster(this._projectId, value, this._isNewCluster)
      .then(() => history.push(`/admin/projects/${this._projectId}/clusters/${value.id}`));
  };

  render() {
    const cluster = this.props.cluster || {};
    const title = this._isNewCluster ? 'New Cluster' : `Cluster ${this._clusterId}`;
    const isLoading = (this._clusterId && isEmpty(cluster));

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={title} />
        {isLoading && <span>Loading...</span>}
        {!isLoading
          && (
          <ClusterForm
            projectId={this._projectId}
            cluster={cluster}
            resolvedCluster={this.props.resolvedCluster}
            isNew={this._isNewCluster}
            onSubmit={this.onSubmit}
            onResolveCluster={this.props.resolveCluster}
            onClearResolvedCluster={this.props.clearResolvedCluster}
          />
          )}
      </div>
    );
  }
}

ClusterEditPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  cluster: PropTypes.object.isRequired,
  resolvedCluster: PropTypes.object.isRequired,
  findCluster: PropTypes.func.isRequired,
  saveCluster: PropTypes.func.isRequired,
  resolveCluster: PropTypes.func.isRequired,
  clearCluster: PropTypes.func.isRequired,
  clearResolvedCluster: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  cluster: state.cluster,
  resolvedCluster: state.resolvedCluster,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findCluster,
  saveCluster,
  clearCluster,
  resolveCluster,
  clearResolvedCluster,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(ClusterEditPage);

export default wrapSubProjectEditPage(connectedPage);
