import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import AlertFormButtons from '../alerts/AlertEditPage/AlertFormButtons';
import FormInput from '../../components/forms/FormInput';
import ApiCallExample from '../../components/ApiCallExample';
import { ClusterHostsRow, mapHostsRows } from './ClusterHostsTable';
import ClusterHostListForm from './ClusterHostListForm';
import ClusterHostUrlForm from './ClusterHostUrlForm';
import ClusterHostConductorForm from './ClusterHostConductorForm';
import ClusterHostConductorTagForm from './ClusterHostConductorTagForm';
import ClusterHostNannyForm from './ClusterHostNannyForm';
import ClusterHostQloudForm from './ClusterHostQloudForm';
import Modal from '../../components/Modal';
import TableViewer from '../../components/TableViewer';
import { TabPane, Tabs } from '../../components/Tabs';
import FormElement from '../../components/forms/FormElement';
import { inputTargetValue } from '../../utils/inputs';
import ClusterHostNetworkForm from './ClusterHostNetworkForm';
import ClusterHostYpForm from './ClusterHostYpForm';
import ClusterHostInstanceGroupForm from './ClusterHostInstanceGroupForm';
import ClusterHostCloudDnsForm from './ClusterHostCloudDnsForm';
import FormCheckbox from '../../components/forms/FormCheckbox';
import ResolvedHostsModal from './ResolvedHostsModal';

const SUB_FORMS = {
  hosts: ClusterHostListForm,
  hostUrls: ClusterHostUrlForm,
  conductorGroups: ClusterHostConductorForm,
  conductorTags: ClusterHostConductorTagForm,
  nannyGroups: ClusterHostNannyForm,
  qloudGroups: ClusterHostQloudForm,
  networks: ClusterHostNetworkForm,
  ypClusters: ClusterHostYpForm,
  instanceGroups: ClusterHostInstanceGroupForm,
  cloudDns: ClusterHostCloudDnsForm,
};

class ClusterForm extends PureComponent {
  static clusterToState = (cluster, isNew) => {
    if (cluster.id) {
      const state = { ...cluster };
      state.id = isNew ? '' : state.id;
      return state;
    }
    return { id: '' };
  };

  constructor(props) {
    super(props);

    const cluster = ClusterForm.clusterToState(props.cluster, props.isNew);
    this.state = {
      cluster,
      clusterHosts: mapHostsRows(cluster),
      isClusterHostFormShown: false,
      selectedClusterHost: null,
    };

    this._nameInputTouched = !!props.cluster.name;
    this._isNewCluster = props.isNew;
  }

  // TODO: make work with state correctly for React 16
  componentDidUpdate(prevProps) {
    // We cannot use setState() because it performs a shallow merge of
    // previous state with the new one. But here we need entirely update state.
    if (this.props.projectId !== prevProps.projectId || this.props.cluster !== prevProps.cluster) {
      const cluster = ClusterForm.clusterToState(this.props.cluster, this.props.isNew);
      this.state = {
        cluster,
        clusterHosts: mapHostsRows(cluster),
        isClusterHostFormShown: false,
      };
    }
  }

  onInputChange = (event) => {
    const { target } = event;
    const value = inputTargetValue(target);

    const cluster = { ...this.state.cluster };
    switch (target.name) {
      case 'id':
        cluster.id = `${this.props.projectId}_${value}`;
        if (!this._nameInputTouched) {
          cluster.name = value;
        }
        break;
      case 'name':
        this._nameInputTouched = true;
        cluster.name = value;
        break;
      default:
        cluster[target.name] = value;
    }

    this.setState({ cluster });
  };

  onSubmit = (event) => {
    event.preventDefault();
    this.props.onSubmit(this.state.cluster);
  };

  onClusterHostClick = (index) => {
    this.setState({ selectedClusterHost: this.state.clusterHosts[index] });
  };

  onClusterHostDelete = (index) => {
    const clusterHost = this.state.clusterHosts[index];
    const cluster = { ...this.state.cluster };
    // eslint-disable-next-line max-len
    cluster[clusterHost.clusterField] = cluster[clusterHost.clusterField].filter((v, i) => i !== clusterHost.index);

    this.setState({
      cluster,
      clusterHosts: mapHostsRows(cluster),
    });
  };

  onAddHost = (event) => {
    event.preventDefault();

    let form = null;
    let field = null;

    /* eslint-disable react/no-string-refs */
    if (this.refs.hostListForm) {
      form = this.refs.hostListForm;
      field = 'hosts';
    } else if (this.refs.hostUrlForm) {
      form = this.refs.hostUrlForm;
      field = 'hostUrls';
    } else if (this.refs.hostConductorForm) {
      form = this.refs.hostConductorForm;
      field = 'conductorGroups';
    } else if (this.refs.hostConductorTagForm) {
      form = this.refs.hostConductorTagForm;
      field = 'conductorTags';
    } else if (this.refs.hostNannyForm) {
      form = this.refs.hostNannyForm;
      field = 'nannyGroups';
    } else if (this.refs.hostQloudForm) {
      form = this.refs.hostQloudForm;
      field = 'qloudGroups';
    } else if (this.refs.hostNetworkForm) {
      form = this.refs.hostNetworkForm;
      field = 'networks';
    } else if (this.refs.hostYpForm) {
      form = this.refs.hostYpForm;
      field = 'ypClusters';
    } else if (this.refs.hostInstanceGroupForm) {
      form = this.refs.hostInstanceGroupForm;
      field = 'instanceGroups';
    } else if (this.refs.hostCloudDnsForm) {
      form = this.refs.hostCloudDnsForm;
      field = 'cloudDns';
    }
    /* eslint-enable */

    if (form && field) {
      const cluster = { ...this.state.cluster };
      const newValue = form.stateToValue();
      if (field in cluster) {
        cluster[field] = [...cluster[field], newValue];
      } else {
        cluster[field] = [newValue];
      }

      this.setState({
        cluster,
        clusterHosts: mapHostsRows(cluster),
        isClusterHostFormShown: false,
      });
    }
  };

  onJsonStateChange = (newState) => {
    this.setState({
      cluster: newState,
      clusterHosts: mapHostsRows(newState),
      isClusterHostFormShown: false,
    });

    this.forceUpdate();
  };

  onResolveCluster = (event) => {
    event.preventDefault();
    this.props.onResolveCluster(this.state.cluster);
  };

  showClusterHostForm = () => {
    this.setState({ isClusterHostFormShown: true });
  };

  hideClusterHostForm = () => {
    this.setState({ isClusterHostFormShown: false });
  };

  hideEditClusterHostForm = () => {
    this.setState({ selectedClusterHost: null });
  };

  render() {
    const { projectId } = this.props;
    const {
      name, port, useFqdn, sensorsTtlDays, tvmDestId,
    } = this.state.cluster;

    let { id } = this.state.cluster;
    if (id.startsWith(`${projectId}_`)) {
      id = id.substring(projectId.length + 1);
    }

    let editModal = null;
    if (this.state.selectedClusterHost) {
      const { type, clusterField, index } = this.state.selectedClusterHost;
      const clusterHost = this.state.cluster[clusterField][index];

      /* eslint-disable react/no-string-refs */
      const Form = SUB_FORMS[clusterField];
      const onOk = () => {
        let { cluster } = this.state;
        const newHosts = [...cluster[clusterField]];
        newHosts[index] = this.refs.hostEditForm.stateToValue();
        cluster = { ...cluster, [clusterField]: newHosts };

        this.setState({
          cluster,
          clusterHosts: mapHostsRows(cluster),
          selectedClusterHost: null,
        });
      };
      editModal = (
        <Modal
          title={`Edit ${type} config`}
          okTitle="Save" onOk={onOk}
          onCancel={this.hideEditClusterHostForm}
          isOpen
        >
          <Form {...clusterHost} ref="hostEditForm" />
        </Modal>
      );
    }
    /* eslint-enable */

    let nameWarning = null;
    if (this._nameInputTouched && !this._isNewCluster) {
      nameWarning = (
        <div>
          <p>
            <strong>
              <span className="glyphicon glyphicon-alert" />
              {' '}
              Important!
            </strong>
          </p>
          <p>
            Modifying the cluster name will change all the URLs and metric queries
            (graphs, dashboards in Solomon, Grafana, Statface, etc.) related to the cluster.
          </p>
          <p>
            If you really want to change the name, please write to us about it:
            {' '}
            <a href="mailto://solomon@yandex-team.ru">solomon@yandex-team.ru</a>
          </p>
        </div>
      );
    }

    const resolvedHostsModal = (
      <ResolvedHostsModal
        resolvedCluster={this.props.resolvedCluster}
        onClearResolvedCluster={this.props.onClearResolvedCluster}
      />
    );

    return (
      <div>
        <form className="form-horizontal">
          <div className="col-lg-6 col-md-7">
            <FormInput
              type="text" name="id" label="Id" value={id} onChange={this.onInputChange}
              prefix={`${projectId}_`}
              help="Unique cluster identificator. Project id will be added as prefix"
              disabled={!this._isNewCluster}
            />
            <FormInput
              type="text" name="name" label="Label name" value={name} onChange={this.onInputChange}
              help="This value will be displayed in selectors in Solomon UI.
                      If you change this value for PUSH cluster make sure to change
                      value of 'cluster' label in your pushing data"
              warning={nameWarning}
            />
            <FormInput
              type="number"
              name="port"
              label="Port"
              value={port}
              onChange={this.onInputChange}
            // eslint-disable-next-line max-len
              help="Port which Solomon shoud use to fetch metrics from your application. Value of this field will override port number defined in service config"
            />
            <FormCheckbox
              name="useFqdn"
              label="Use FQDN"
              value={useFqdn}
              onChange={this.onInputChange}
            // eslint-disable-next-line max-len
              help={<span>If true Solomon will write FQDN to host label instead of short host</span>}
            />
            <FormInput
              type="number"
              name="sensorsTtlDays"
              label="Metrics TTL"
              value={sensorsTtlDays}
              onChange={this.onInputChange}
              suffix="days"
            // eslint-disable-next-line max-len
              help="If this field is filled, Solomon will drop all metrics where last point was metrics TTL days ago"
            />
            <FormInput
              type="text"
              name="tvmDestId"
              label="TVM destination id"
              value={tvmDestId}
              onChange={this.onInputChange}
              help="TVM clent id of your application that Solomon will contact to download metrics from"
            />
            <FormElement label="Cluster hosts">
              <TableViewer
                columns={['#', 'Type', 'Value', 'Labels', '']}
                values={this.state.clusterHosts}
                row={ClusterHostsRow}
                limit={15}
                onAdd={this.showClusterHostForm}
                onRowClick={this.onClusterHostClick}
                onDelete={this.onClusterHostDelete}
              />
            </FormElement>
            <AlertFormButtons onSubmit={this.onSubmit} onExplain={this.onResolveCluster} />
          </div>
          <div className="col-lg-6 col-md-5">
            <ApiCallExample
              path={`/projects/${projectId}/clusters`}
              value={this.state.cluster}
              isNew={this._isNewCluster}
              onChange={this.onJsonStateChange}
            />
          </div>
        </form>
        <Modal
          title="Add hosts to cluster"
          okTitle="Add" onOk={this.onAddHost}
          onCancel={this.hideClusterHostForm}
          isOpen={this.state.isClusterHostFormShown}
        >
          <Tabs vertical>
            {/* eslint-disable react/no-string-refs */}
            <TabPane label="Hosts">
              <ClusterHostListForm ref="hostListForm" />
            </TabPane>
            <TabPane label="Host URLs">
              <ClusterHostUrlForm ref="hostUrlForm" />
            </TabPane>
            <TabPane label="Conductor">
              <ClusterHostConductorForm ref="hostConductorForm" />
            </TabPane>
            <TabPane label="Conductor tag">
              <ClusterHostConductorTagForm ref="hostConductorTagForm" />
            </TabPane>
            <TabPane label="Nanny">
              <ClusterHostNannyForm ref="hostNannyForm" />
            </TabPane>
            <TabPane label="Qloud">
              <ClusterHostQloudForm ref="hostQloudForm" />
            </TabPane>
            <TabPane label="Network">
              <ClusterHostNetworkForm ref="hostNetworkForm" />
            </TabPane>
            <TabPane label="YP clusters">
              <ClusterHostYpForm ref="hostYpForm" />
            </TabPane>
            <TabPane label="Instance group">
              <ClusterHostInstanceGroupForm ref="hostInstanceGroupForm" />
            </TabPane>
            <TabPane label="Cloud DNS">
              <ClusterHostCloudDnsForm ref="hostCloudDnsForm" />
            </TabPane>
          </Tabs>
        </Modal>
        {editModal}
        {resolvedHostsModal}
      </div>
    );
  }
}

ClusterForm.propTypes = {
  projectId: PropTypes.string.isRequired,
  cluster: PropTypes.object.isRequired,
  resolvedCluster: PropTypes.object.isRequired,
  isNew: PropTypes.bool.isRequired,
  onSubmit: PropTypes.func.isRequired,
  onResolveCluster: PropTypes.func.isRequired,
  onClearResolvedCluster: PropTypes.func.isRequired,
};

export default ClusterForm;
