import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import FormInput from '../../components/forms/FormInput';
import { strToLabels, labelsToStr } from '../../utils/labels';
import { inputTargetValue } from '../../utils/inputs';
import LabelsInput from './LabelsInput';

class ClusterHostListForm extends PureComponent {
  constructor(props) {
    super(props);
    this.state = {
      urlPattern: props.urlPattern,
      ranges: props.ranges,
      dc: props.dc,
      labels: labelsToStr(props.labels),
    };
    if (props.onSubmit) {
      this.onSubmit = (event) => {
        event.preventDefault();
        props.onSubmit(this.stateToValue());
      };
    }
  }

  onInputChange = (event) => {
    const { target } = event;
    const { name } = target;
    const value = inputTargetValue(target);
    this.setState({ [name]: value });
  };

  stateToValue() {
    const {
      urlPattern, ranges, dc, labels,
    } = this.state;
    return {
      urlPattern,
      ranges,
      dc,
      labels: strToLabels(labels),
    };
  }

  render() {
    const {
      urlPattern, ranges, dc, labels,
    } = this.state;
    return (
      <form className="form-horizontal" onSubmit={this.onSubmit}>
        <FormInput
          type="text"
          name="urlPattern"
          label="URL pattern"
          value={urlPattern}
          onChange={this.onInputChange}
          help="Don't mind URL in field name, it is actually URL pattern
                and may contain printf-like pattern (e.g. '%d' or '%03d'
                that is filled with ranges"
        />
        <FormInput
          type="text"
          name="ranges"
          label="Pattern ranges"
          value={ranges}
          onChange={this.onInputChange}
          help="One or more ranges deilimted by space to fill above pattern
                (e.g. '0-10 20-40')"
        />
        <FormInput
          type="text"
          name="dc"
          label="DC"
          value={dc}
          onChange={this.onInputChange}
          help="One or data center name"
        />
        <LabelsInput
          value={labels}
          onChange={this.onInputChange}
        />
      </form>
    );
  }
}

ClusterHostListForm.propTypes = {
  onSubmit: PropTypes.func,
  urlPattern: PropTypes.string,
  ranges: PropTypes.string,
  dc: PropTypes.string,
  labels: PropTypes.arrayOf(PropTypes.string),
};

ClusterHostListForm.defaultProps = {
  onSubmit: null,
  urlPattern: '',
  ranges: '',
  dc: '',
  labels: [],
};

export default ClusterHostListForm;
