import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import FormInput from '../../components/forms/FormInput';
import FormCheckbox from '../../components/forms/FormCheckbox';
import { strToLabels, labelsToStr } from '../../utils/labels';
import { inputTargetValue } from '../../utils/inputs';
import FormElement from '../../components/forms/FormElement';
import RadioButtons from '../../components/RadioButtons';
import LabelsInput from './LabelsInput';

const NANNY_ENVS = [
  'PRODUCTION',
  'ADMIN',
];

class ClusterHostNannyForm extends PureComponent {
  constructor(props) {
    super(props);
    this.state = {
      service: props.service,
      useFetchedPort: props.useFetchedPort,
      env: props.env,
      portShift: props.portShift,
      cfgGroup: labelsToStr(props.cfgGroup),
      labels: labelsToStr(props.labels),
    };
    if (props.onSubmit) {
      this.onSubmit = (event) => {
        event.preventDefault();
        props.onSubmit(this.stateToValue());
      };
    }
  }

  onInputChange = (event) => {
    const { target } = event;
    const { name } = target;
    const value = inputTargetValue(target);
    this.setState({ [name]: value });
  };

  onEnvSelect = (env) => {
    this.setState({ env });
  };

  stateToValue() {
    const {
      service, useFetchedPort, env, portShift, cfgGroup, labels,
    } = this.state;
    return {
      service,
      useFetchedPort,
      env,
      portShift,
      cfgGroup: strToLabels(cfgGroup),
      labels: strToLabels(labels),
    };
  }

  render() {
    const {
      service, useFetchedPort, env, portShift, cfgGroup, labels,
    } = this.state;
    return (
      <form className="form-horizontal" onSubmit={this.onSubmit}>
        <FormInput
          type="text"
          name="service"
          label="Group name"
          value={service}
          onChange={this.onInputChange}
          help="Name of Nanny service as in /v2/services/{service}/current_state/instances/
                path"
        />
        <FormCheckbox
          name="useFetchedPort"
          label="Use fetched port"
          value={useFetchedPort}
          onChange={this.onInputChange}
          help="Use service port as configured in Nanny service"
        />
        <FormElement
          label="Environment"
          help="Use Nanny admin installation"
        >
          <RadioButtons
            choices={NANNY_ENVS} selected={env} onSelect={this.onEnvSelect}
          />
        </FormElement>
        {useFetchedPort && (
          <FormInput
            type="number"
            name="portShift"
            label="Port shift"
            value={portShift}
            onChange={this.onInputChange}
            help="This value will be added to port before establishing connection
                  with your service"
          />
        )}
        <FormInput
          type="text"
          name="cfgGroup"
          label="Config groups"
          value={cfgGroup}
          onChange={this.onInputChange}
          help="One or more config groups for cfg-api
                http://api.gencfg.yandex-team.ru/{tag}/groups/{cfgGroup}.
                Groups must be delimited by space"
        />
        <LabelsInput
          value={labels}
          onChange={this.onInputChange}
        />
      </form>
    );
  }
}

ClusterHostNannyForm.propTypes = {
  onSubmit: PropTypes.func,
  service: PropTypes.string,
  useFetchedPort: PropTypes.bool,
  env: PropTypes.string,
  portShift: PropTypes.number,
  cfgGroup: PropTypes.array,
  labels: PropTypes.arrayOf(PropTypes.string),
};

ClusterHostNannyForm.defaultProps = {
  onSubmit: null,
  service: '',
  useFetchedPort: false,
  env: 'PRODUCTION',
  portShift: 0,
  cfgGroup: [],
  labels: [],
};

export default ClusterHostNannyForm;
