import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import FormInput from '../../components/forms/FormInput';
import { strToLabels, labelsToStr } from '../../utils/labels';
import { inputTargetValue } from '../../utils/inputs';
import LabelsInput from './LabelsInput';

class ClusterHostNetworkForm extends PureComponent {
  constructor(props) {
    super(props);
    this.state = {
      network: props.network,
      port: props.port,
      labels: labelsToStr(props.labels),
    };
    if (props.onSubmit) {
      this.onSubmit = (event) => {
        event.preventDefault();
        props.onSubmit(this.stateToValue());
      };
    }
  }

  onInputChange = (event) => {
    const { target } = event;
    const { name } = target;
    const value = inputTargetValue(target);
    this.setState({ [name]: value });
  };

  stateToValue() {
    const { network, port, labels } = this.state;
    return {
      network,
      port,
      labels: strToLabels(labels),
    };
  }

  render() {
    const { network, port, labels } = this.state;
    return (
      <form className="form-horizontal" onSubmit={this.onSubmit}>
        <FormInput
          type="text"
          name="network"
          label="Network"
          value={network}
          onChange={this.onInputChange}
          help="Single IP address (v4/v6) or subnetwork expressed in Classless Inter-Domain Routing (CIDR) notation"
        />
        <FormInput
          type="number"
          name="port"
          label="Port"
          value={port}
          onChange={this.onInputChange}
          help="Port which Solomon shoud use to fetch metrics (optional)"
        />
        <LabelsInput
          value={labels}
          onChange={this.onInputChange}
        />
      </form>
    );
  }
}

ClusterHostNetworkForm.propTypes = {
  network: PropTypes.string,
  port: PropTypes.number,
  labels: PropTypes.arrayOf(PropTypes.string),
  onSubmit: PropTypes.func,
};

ClusterHostNetworkForm.defaultProps = {
  onSubmit: null,
  network: '',
  port: 0,
  labels: [],
};

export default ClusterHostNetworkForm;
