import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import FormInput from '../../components/forms/FormInput';
import { strToLabels, labelsToStr } from '../../utils/labels';
import { inputTargetValue } from '../../utils/inputs';
import FormSelect from '../../components/forms/FormSelect';
import LabelsInput from './LabelsInput';

const DEPLOYMENT_OPTIONS = [
  { title: 'Test', value: 'test' },
  { title: 'Prestable', value: 'prestable' },
  { title: 'Stable', value: 'stable' },
];

class ClusterHostQloudForm extends PureComponent {
  constructor(props) {
    super(props);
    this.state = {
      component: props.component,
      environment: props.environment,
      application: props.application,
      project: props.project,
      deployment: props.deployment,
      labels: labelsToStr(props.labels),
    };
    if (props.onSubmit) {
      this.onSubmit = (event) => {
        event.preventDefault();
        props.onSubmit(this.stateToValue());
      };
    }
  }

  onInputChange = (event) => {
    const { target } = event;
    const { name } = target;
    const value = inputTargetValue(target);
    this.setState({ [name]: value });
  };

  onSelectChange = (event) => {
    this.setState({ [event.target.name]: event.target.value });
  };

  stateToValue() {
    const {
      component, environment, application, project, deployment, labels,
    } = this.state;
    return {
      component,
      environment,
      application,
      project,
      deployment,
      labels: strToLabels(labels),
    };
  }

  render() {
    const {
      component, environment, application, project, deployment, labels,
    } = this.state;
    return (
      <form className="form-horizontal" onSubmit={this.onSubmit}>
        <p>
          <a
            href="https://wiki.yandex-team.ru/qloud/doc/environment/component/dns-discovery/"
            target="_blank"
            rel="noopener noreferrer"
          >
            Read more about Qloud DNS Discovery
          </a>
        </p>
        <FormInput
          type="text"
          name="component"
          label="Component"
          value={component}
          onChange={this.onInputChange}
        />
        <FormInput
          type="text"
          name="environment"
          label="Environment"
          value={environment}
          onChange={this.onInputChange}
        />
        <FormInput
          type="text"
          name="application"
          label="Application"
          value={application}
          onChange={this.onInputChange}
        />
        <FormInput
          type="text"
          name="project"
          label="Project"
          value={project}
          onChange={this.onInputChange}
        />
        <FormSelect
          name="deployment"
          label="Deployment"
          options={DEPLOYMENT_OPTIONS}
          value={deployment}
          defaultValue="stable"
          onChange={this.onSelectChange}
        />
        <LabelsInput
          value={labels}
          onChange={this.onInputChange}
        />
      </form>
    );
  }
}

ClusterHostQloudForm.propTypes = {
  onSubmit: PropTypes.func,
  component: PropTypes.string,
  environment: PropTypes.string,
  application: PropTypes.string,
  project: PropTypes.string,
  deployment: PropTypes.string,
  labels: PropTypes.arrayOf(PropTypes.string),
};

ClusterHostQloudForm.defaultProps = {
  onSubmit: null,
  component: '',
  environment: '',
  application: '',
  project: '',
  deployment: 'stable',
  labels: [],
};

export default ClusterHostQloudForm;
