import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import FormCheckbox from '../../components/forms/FormCheckbox';
import FormInput from '../../components/forms/FormInput';
import { strToLabels, labelsToStr } from '../../utils/labels';
import { inputTargetValue } from '../../utils/inputs';
import LabelsInput from './LabelsInput';

class ClusterHostUrlForm extends PureComponent {
  constructor(props) {
    super(props);
    this.state = {
      url: props.url,
      ignorePorts: props.ignorePorts,
      labels: labelsToStr(props.labels),
    };
    if (props.onSubmit) {
      this.onSubmit = (event) => {
        event.preventDefault();
        props.onSubmit(this.stateToValue());
      };
    }
  }

  onInputChange = (event) => {
    const { target } = event;
    const { name } = target;
    const value = inputTargetValue(target);
    this.setState({ [name]: value });
  };

  stateToValue() {
    const { url, ignorePorts, labels } = this.state;
    return {
      url,
      ignorePorts,
      labels: strToLabels(labels),
    };
  }

  render() {
    const { url, ignorePorts, labels } = this.state;
    return (
      <form className="form-horizontal" onSubmit={this.onSubmit}>
        <FormInput
          type="text"
          name="url"
          label="URL"
          value={url}
          onChange={this.onInputChange}
          help="URL of text file with host list"
        />
        <FormCheckbox
          name="ignorePorts"
          label="Ignore ports"
          value={ignorePorts}
          onChange={this.onInputChange}
          help="If enabled then ports present in downloaded file will be ignored"
        />
        <LabelsInput
          value={labels}
          onChange={this.onInputChange}
        />
      </form>
    );
  }
}

ClusterHostUrlForm.propTypes = {
  onSubmit: PropTypes.func,
  url: PropTypes.string,
  ignorePorts: PropTypes.bool,
  labels: PropTypes.arrayOf(PropTypes.string),
};

ClusterHostUrlForm.defaultProps = {
  onSubmit: null,
  url: '',
  ignorePorts: false,
  labels: [],
};

export default ClusterHostUrlForm;
