import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import FormInput from '../../components/forms/FormInput';
import { strToLabels, labelsToStr } from '../../utils/labels';
import { inputTargetValue } from '../../utils/inputs';
import FormDatalist from '../../components/forms/FormDatalist';
import LabelsInput from './LabelsInput';

const CLUSTERS_DATALIST = [
  'man',
  'sas',
  'vla',
  'sas-test',
  'man-pre',
  'myt',
  'iva',
  'xdc',
];

class ClusterHostYpForm extends PureComponent {
  constructor(props) {
    super(props);
    this.state = {
      ypLabel: props.ypLabel,
      tvmLabel: props.tvmLabel,
      podSetId: props.podSetId,
      endpointSetId: props.endpointSetId,
      cluster: props.cluster,
      labels: labelsToStr(props.labels),
    };
    if (props.onSubmit) {
      this.onSubmit = (event) => {
        event.preventDefault();
        props.onSubmit(this.stateToValue());
      };
    }
  }

  onInputChange = (event) => {
    const { target } = event;
    const { name } = target;
    const value = inputTargetValue(target);
    this.setState({ [name]: value });
  };

  stateToValue() {
    const {
      ypLabel, tvmLabel, podSetId, endpointSetId, cluster, labels,
    } = this.state;
    return {
      ypLabel,
      tvmLabel,
      podSetId,
      endpointSetId,
      cluster,
      labels: strToLabels(labels),
    };
  }

  render() {
    const {
      ypLabel, tvmLabel, podSetId, endpointSetId, cluster, labels,
    } = this.state;
    return (
      <form className="form-horizontal" onSubmit={this.onSubmit}>
        <p>Exactly one of the podSetId, endpoints or tvmLabel must be specified</p>
        <FormInput
          type="text"
          name="podSetId"
          label="Pod set ID"
          value={podSetId}
          onChange={this.onInputChange}
        />
        <FormInput
          type="text"
          name="endpointSetId"
          label="Endpoint set ID"
          value={endpointSetId}
          onChange={this.onInputChange}
        />
        <FormInput
          type="text"
          name="ypLabel"
          label="YP label"
          help="Discover pods by picking only the ones that have the specified label value set to true
              (i.e. $YP_LABEL = %true) from any pod set"
          value={ypLabel}
          onChange={this.onInputChange}
        />
        <FormInput
          type="text"
          name="tvmLabel"
          label="YP TVM label"
          help="Obtain a TVM id for the application from the specified YP label"
          value={tvmLabel}
          onChange={this.onInputChange}
        />
        <FormDatalist
          type="text"
          name="cluster"
          label="Cluster"
          value={cluster}
          datalist={CLUSTERS_DATALIST}
          onChange={this.onInputChange}
        />
        <LabelsInput
          value={labels}
          onChange={this.onInputChange}
        />
      </form>
    );
  }
}

ClusterHostYpForm.propTypes = {
  ypLabel: PropTypes.string,
  tvmLabel: PropTypes.string,
  podSetId: PropTypes.string,
  endpointSetId: PropTypes.string,
  cluster: PropTypes.string,
  labels: PropTypes.arrayOf(PropTypes.string),
  onSubmit: PropTypes.func,
};

ClusterHostYpForm.defaultProps = {
  onSubmit: null,
  ypLabel: '',
  tvmLabel: '',
  podSetId: '',
  endpointSetId: '',
  cluster: '',
  labels: [],
};

export default ClusterHostYpForm;
