import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import LabelsList from '../../components/LabelsList';
import TableViewer from '../../components/TableViewer';

const mapper = (type, style, clusterField, valueField) => (value, index) => ({
  type,
  style,
  clusterField,
  index,
  value: value[valueField],
  labels: value.labels || [],
});

const ypClusterMapper = () => (value, index) => ({
  type: 'YP cluster',
  style: 'success',
  clusterField: 'ypClusters',
  index,
  value: value.podSetId || value.endpointSetId,
  labels: value.labels || [],
});

const NannyMapper = () => (value, index) => ({
  type: value.env === 'ADMIN' ? 'Admin Nanny' : 'Nanny',
  style: 'info',
  clusterField: 'nannyGroups',
  index,
  value: value.service,
  labels: value.labels || [],
});

const MAPPERS = {
  hosts: mapper('Host', 'default', 'hosts', 'urlPattern'),
  hostUrls: mapper('Host URL', 'primary', 'hostUrls', 'url'),
  conductorGroups: mapper('Conductor', 'success', 'conductorGroups', 'group'),
  conductorTags: mapper('Conductor tag', 'success', 'conductorTags', 'name'),
  nannyGroups: NannyMapper(),
  qloudGroups: mapper('Qloud', 'warning', 'qloudGroups', 'application'),
  networks: mapper('Network', 'info', 'networks', 'network'),
  ypClusters: ypClusterMapper(),
  instanceGroups: mapper('Instance group', 'info', 'instanceGroups', 'instanceGroupId'),
  cloudDns: mapper('Cloud DNS', 'info', 'cloudDns', 'name'),
};

export const mapHostsRows = (cluster) => {
  const rows = [];

  Object.keys(MAPPERS).forEach((field) => {
    if (field in cluster) {
      rows.push(...cluster[field].map(MAPPERS[field]));
    }
  });
  return rows;
};

export const isEmptyClusterHosts = (cluster) => {
  let isEmpty = true;
  Object.keys(MAPPERS).forEach((field) => {
    if (field in cluster && cluster[field].length > 0) {
      isEmpty = false;
    }
  });
  return isEmpty;
};

function wrapHandler(index, handler) {
  if (!handler) {
    return null;
  }
  return (event) => {
    event.preventDefault();
    event.stopPropagation();
    handler(index);
  };
}

export class ClusterHostsRow extends PureComponent {
  constructor(props) {
    super(props);
    this.onClick = wrapHandler(props.index, props.onClick);
    this.onDelete = wrapHandler(props.index, props.onDelete);
  }

  render() {
    const { index, value } = this.props;
    const labelClassName = `label label-${value.style} label-item-type`;
    return (
      <tr onClick={this.onClick}>
        <td>{index + 1}</td>
        <td><span className={labelClassName}>{value.type}</span></td>
        <td>{value.value}</td>
        <td><LabelsList values={value.labels} type="info" /></td>
        {this.onDelete
          && (
          <td className="actions">
            <button type="button" onClick={this.onDelete}>
              <i className="glyphicon glyphicon-remove" />
            </button>
          </td>
          )}
      </tr>
    );
  }
}

ClusterHostsRow.propTypes = {
  index: PropTypes.number.isRequired,
  value: PropTypes.shape({
    style: PropTypes.string,
    type: PropTypes.string,
    title: PropTypes.string,
    value: PropTypes.string,
    labels: PropTypes.arrayOf(PropTypes.string),
  }).isRequired,
  onClick: PropTypes.func,
  onDelete: PropTypes.func,
};

ClusterHostsRow.defaultProps = {
  onClick: null,
  onDelete: null,
};

const ClusterHostsTable = ({ cluster, ...props }) => (
  <TableViewer
    columns={['#', 'Type', 'Value', 'Labels', '']}
    values={mapHostsRows(cluster)}
    row={ClusterHostsRow}
    {...props}
  />
);

ClusterHostsTable.propTypes = {
  cluster: PropTypes.object.isRequired,
};

export default ClusterHostsTable;
