import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';
import isEmpty from 'lodash/isEmpty';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import LabeledValue from '../../components/LabeledValue';
import Panel from '../../components/Panel';
import ShowActions from '../../components/ShowActions';

import ClusterHostsTable, { isEmptyClusterHosts } from './ClusterHostsTable';

import { isDefaultItemState } from '../../utils/itemState';
import ShardModalForm from './ShardModalForm';
import Modal from '../../components/Modal';
import { TabPane, Tabs } from '../../components/Tabs';
import EntityInfo from '../../components/EntityInfo';
import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';
import { PROJECT_PERMISSIONS } from '../../auth/ProjectPermissions';
import AssociatedEntitiesTable from '../../components/AssociatedEntitiesTable';
import ConfirmDeletionModal from '../../components/ShowActions/ConfirmDeletionModal';

import { findCluster, deleteCluster } from '../../store/reducers/clusters/cluster';
import { findClusterServices } from '../../store/reducers/clusters/clusterServices';
import { findProjectServices } from '../../store/reducers/services/servicesPage';
import {
  createShard, deleteShard, fetchShard, updateShard,
} from '../../api/shards';
import { clearResolvedCluster, resolveCluster } from '../../store/reducers/clusters/resolvedCluster';
import ResolvedHostsModal from './ResolvedHostsModal';

class ClusterPage extends PureComponent {
  constructor(props) {
    super(props);
    const { projectId, clusterId } = props.match.params;
    this._projectId = projectId;
    this._clusterId = clusterId;
    this.state = {
      newShardModalShown: false,
      deleteShardModalShown: false,
      errorMessage: '',
      askAboutShardDeletion: true,
      deletedShardId: '',
    };
  }

  componentDidMount() {
    this.props.findCluster(this._projectId, this._clusterId);
    this.props.findClusterServices(this._projectId, this._clusterId);
    this.props.findProjectServices(this._projectId, { pageSize: 'all' });
  }

  componentDidUpdate(prevProps) {
    const { params } = this.props.match;
    const prevParams = prevProps.match.params;
    if (params.clusterId !== prevParams.clusterId) {
      this._projectId = params.projectId;
      this._clusterId = params.clusterId;
      this.props.findCluster(this._projectId, this._clusterId);
      this.props.findClusterServices(this._projectId, this._clusterId);
      this.props.findProjectServices(this._projectId, { pageSize: 'all' });
    }
  }

  onDelete = (event) => {
    event.preventDefault();
    const { history } = this.props;
    this.props.deleteCluster(this._projectId, this._clusterId)
      .then((action) => {
        if (action) {
          history.push(`/admin/projects/${this._projectId}/clusters`);
        }
      });
  };

  onAddNewShard = () => {
    /* eslint-disable react/no-string-refs */
    const state = this.refs.shardModalForm.stateToValue();
    createShard(this._projectId, state)
      .catch((error) => this.setState({ errorMessage: error.message }))
      .then(this.onHideNewShardModal)
      .then(() => this.props.findClusterServices(this._projectId, this._clusterId));
  };

  onDeactivateShard = (association) => {
    const { shardId } = association;
    fetchShard(this._projectId, shardId)
      .then((shard) => updateShard(this._projectId, { ...shard, state: 'INACTIVE' }))
      .then(() => this.props.findClusterServices(this._projectId, this._clusterId));
  };

  onActivateShard = (association) => {
    const { shardId } = association;
    fetchShard(this._projectId, shardId)
      .then((shard) => updateShard(this._projectId, { ...shard, state: 'RW' }))
      .then(() => this.props.findClusterServices(this._projectId, this._clusterId));
  };

  onDeleteShard = (shardId) => deleteShard(this._projectId, shardId).then(() => {
    this.props.findClusterServices(this._projectId, this._clusterId);
  });

  onShowNewShardModal = () => {
    this.setState({ newShardModalShown: true });
  };

  onHideNewShardModal = () => {
    this.setState({ newShardModalShown: false });
  };

  onConfirmDeletionOrDeleteShard = (association) => {
    const { shardId } = association;
    if (this.state.askAboutShardDeletion) {
      this.setState({ deleteShardModalShown: true, deletedShardId: shardId });
    } else {
      this.onDeleteShard(shardId);
    }
  };

  onConfirmShardDeletion = (event, dontAskAnymore) => {
    const shardId = this.state.deletedShardId;
    this.setState({ deleteShardModalShown: false, askAboutShardDeletion: !dontAskAnymore, deletedShardId: '' }, () => {
      this.onDeleteShard(shardId);
    });
  };

  onHideDeleteShardModal = () => {
    this.setState({ deleteShardModalShown: false, deletedShardId: '' });
  };

  onResolveCluster = () => {
    this.props.resolveCluster(this.props.cluster);
  };

  onClearResolvedCluster = () => {
    this.props.clearResolvedCluster();
  };

  _clusterContent = (cluster) => {
    if (isEmpty(cluster)) {
      return <span>Loading...</span>;
    }
    return (
      <div>
        <Panel title="Generic options">
          <LabeledValue label="Label name" value={cluster.name} />
          {cluster.port
            ? <LabeledValue label="Port" value={cluster.port} hint="overrides port from service" />
            : null}
          {cluster.useFqdn
            ? <LabeledValue label="Use FQDN" value="Yes" />
            : null}
          {cluster.sensorsTtlDays
            ? <LabeledValue label="Metrics TTL" value={`${cluster.sensorsTtlDays} days`} hint="overrides metrics TTL from service" />
            : null}
          {!!cluster.tvmDestId && (
            <LabeledValue
              label="TVM destination id"
              value={cluster.tvmDestId}
            />
          )}
          <EntityInfo entity={cluster} />
        </Panel>
        {!isEmptyClusterHosts(cluster) && (
          <Panel title="Cluster hosts">
            <ClusterHostsTable cluster={cluster} limit={5} />
            <div style={{ marginTop: '10px' }}>
              <button type="button" className="btn btn-default" onClick={this.onResolveCluster}>
                Resolve
              </button>
              <ResolvedHostsModal
                resolvedCluster={this.props.resolvedCluster}
                onClearResolvedCluster={this.onClearResolvedCluster}
              />
            </div>
          </Panel>
        )}
      </div>
    );
  };

  render() {
    const {
      cluster, clusterServices, servicesPage, projectAuth,
    } = this.props;

    const activeClusterServices = clusterServices.filter((c) => isDefaultItemState(c.state));
    const inactiveClusterServices = clusterServices.filter((c) => !isDefaultItemState(c.state));

    const clusterServicesIds = clusterServices.map((s) => s.id);
    const allClusterIds = (servicesPage.result || []).map((s) => s.id);
    const otherServiceIds = allClusterIds.filter((s) => clusterServicesIds.indexOf(s) < 0);

    const canEdit = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_UPDATE);
    const canDelete = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_DELETE);

    const canAddShard = canEdit && otherServiceIds.length > 0;

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Cluster ${this._clusterId}`} />
        <div className="row">
          <div className="col-xs-6">
            {this._clusterContent(cluster)}
            <ShowActions
              editPath={`/admin/projects/${this._projectId}/clusters/${this._clusterId}/edit`}
              duplicatePath={`/admin/projects/${this._projectId}/clusters/new?duplicateOf=${this._clusterId}`}
              onDelete={this.onDelete}
              canEdit={canEdit}
              canChangeState={false}
              canDelete={canDelete && isEmpty(clusterServices)}
            />
          </div>
          <div className="col-xs-6">
            <Panel title="Associated services">
              <Tabs>
                <TabPane label="Active">
                  <div>
                    <AssociatedEntitiesTable
                      type="service"
                      projectId={this._projectId}
                      values={activeClusterServices}
                      limit={5}
                      active
                      onAdd={canAddShard ? this.onShowNewShardModal : null}
                      onToggle={canEdit ? this.onDeactivateShard : null}
                      onDelete={canDelete ? this.onConfirmDeletionOrDeleteShard : null}
                    />
                    <Modal
                      title="Add new shard"
                      okTitle="Save"
                      onOk={this.onAddNewShard}
                      onCancel={this.onHideNewShardModal}
                      isOpen={this.state.newShardModalShown}
                    >
                      <ShardModalForm
                        ref="shardModalForm"
                        projectId={this._projectId}
                        clusterId={this._clusterId}
                        services={otherServiceIds}
                        errorMessage={this.state.errorMessage}
                        onSubmit={this.onAddNewShard}
                      />
                    </Modal>
                  </div>
                </TabPane>
                <TabPane label="Inactive">
                  <AssociatedEntitiesTable
                    type="service"
                    projectId={this._projectId}
                    values={inactiveClusterServices}
                    active={false}
                    limit={5}
                    onToggle={canEdit ? this.onActivateShard : null}
                    onDelete={canDelete ? this.onConfirmDeletionOrDeleteShard : null}
                  />
                </TabPane>
              </Tabs>
              <ConfirmDeletionModal
                isOpen={this.state.deleteShardModalShown}
                showDontAskAnymore={this.state.askAboutShardDeletion}
                onConfirm={this.onConfirmShardDeletion}
                onCancel={this.onHideDeleteShardModal}
              />
            </Panel>
          </div>
        </div>
      </div>
    );
  }
}

ClusterPage.propTypes = {
  match: PropTypes.object.isRequired,
  projectAuth: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  cluster: PropTypes.object.isRequired,
  clusterServices: PropTypes.array.isRequired,
  servicesPage: PropTypes.object.isRequired,
  resolvedCluster: PropTypes.object.isRequired,
  findCluster: PropTypes.func.isRequired,
  findClusterServices: PropTypes.func.isRequired,
  findProjectServices: PropTypes.func.isRequired,
  deleteCluster: PropTypes.func.isRequired,
  resolveCluster: PropTypes.func.isRequired,
  clearResolvedCluster: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  cluster: state.cluster,
  servicesPage: state.servicesPage,
  clusterServices: state.clusterServices,
  resolvedCluster: state.resolvedCluster,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findCluster,
  deleteCluster,
  findClusterServices,
  findProjectServices,
  resolveCluster,
  clearResolvedCluster,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(ClusterPage);

export default wrapSubProjectReadPage(connectedPage);
