import React, { PureComponent } from 'react';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import TableToolbar from '../../components/TableToolbar';
import Pagination from '../../components/Pagination';
import { formatSearch, parseSearch } from '../../utils/url';
import Auth from '../../auth/Auth';
import { ADMIN } from '../../auth/Roles';

import { findGlobalClusters, clearGlobalClusters } from '../../store/reducers/clusters/globalClustersPage';

const mapSearchToState = (search) => {
  const params = parseSearch(search);
  return {
    text: params.get('text') || '',
    page: params.get('page') || 0,
    pageSize: params.get('pageSize') || '',
  };
};

const mapStateToSearch = (state) => `?${formatSearch(state)}`;

const cleanState = (state) => {
  const params = {};
  const { text, page, pageSize } = state;
  if (text) {
    params.text = text;
  }
  if (page !== 0) {
    params.page = page;
  }
  if (pageSize !== '' && pageSize !== 30) {
    params.pageSize = pageSize;
  }
  return params;
};

class GlobalClustersPage extends PureComponent {
  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
    this.state = mapSearchToState(props.location.search);
  }

  componentDidMount() {
    this.props.findGlobalClusters(cleanState(this.state));
  }

  componentWillUnmount() {
    this.props.clearGlobalClusters();
  }

  onFilterChange = (event) => {
    const text = event.target.value;
    this.doReload({ ...this.state, text, page: 0 });
  };

  onActivePageChange = (activePage) => {
    this.doReload({ ...this.state, page: activePage - 1 });
  };

  onPageSizeChange = (pageSize) => {
    this.doReload({ ...this.state, pageSize, page: 0 });
  };

  doReload(newState) {
    this.setState(newState);
    const newCleanedState = cleanState(newState);
    this.props.history.replace(mapStateToSearch(newCleanedState));
    this.props.findGlobalClusters(newCleanedState);
  }

  render() {
    if (!Auth.hasRole(ADMIN)) {
      return (
        <div className="alert alert-danger">
          You have no permissions to see this page
        </div>
      );
    }

    const { clustersPage } = this.props;

    let body = null;

    if (clustersPage.data) {
      const { result, page } = clustersPage.data;
      const indexOffset = (page !== undefined)
        ? page.pageSize * page.current
        : 0;

      body = (
        <>
          <table className="table table-condensed table-hover">
            <thead>
              <tr>
                <th>#</th>
                <th>Id</th>
                <th>Project</th>
                <th>Name</th>
              </tr>
            </thead>
            <tbody>
              {result.map((c, i) => (
                <tr key={c.id}>
                  <td>{i + indexOffset + 1}</td>
                  <td><Link to={`/admin/projects/${c.projectId}/clusters/${c.id}`}>{c.id}</Link></td>
                  <td>
                    <Link to={`/admin/projects/${c.projectId}`}>{c.projectId}</Link>
                  </td>
                  <td>{c.name}</td>
                </tr>
              ))}
            </tbody>
          </table>
          <Pagination
            activePage={page.current + 1}
            pageCount={page.pagesCount}
            pageSize={page.pageSize}
            totalCount={page.totalCount}
            onActivePageChange={this.onActivePageChange}
            onPageSizeChange={this.onPageSizeChange}
          />
        </>
      );
    } else if (clustersPage.loading) {
      body = <div>Loading...</div>;
    }

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title="Clusters" />
        <TableToolbar
          filter={this.state.text}
          onFilterChange={this.onFilterChange}
        />
        {body}
      </div>
    );
  }
}

GlobalClustersPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  clustersPage: PropTypes.object.isRequired,
  findGlobalClusters: PropTypes.func.isRequired,
  clearGlobalClusters: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  clustersPage: state.globalClustersPage,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findGlobalClusters,
  clearGlobalClusters,
}, dispatch);

export default connect(mapStateToProps, mapDispatchToProps)(GlobalClustersPage);
