import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';
import isEmpty from 'lodash/isEmpty';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import DashboardForm from './DashboardForm';

import { parseSearch } from '../../utils/url';

import { wrapSubProjectEditPage } from '../projects/wrapSubProjectPage';

import { findDashboard, saveDashboard, clearDashboard } from '../../store/reducers/dashboards/dashboard';
import NewUiAlert from '../../components/NewUiAlert';

class DashboardEditPage extends PureComponent {
  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
    this._dashboardId = props.match.params.dashboardId;
    this._isNew = !this._dashboardId;

    if (this._isNew) {
      this._dashboardId = parseSearch(props.location.search).get('duplicateOf');
    }
  }

  componentDidMount() {
    const { dashboard } = this.props;
    if (this._dashboardId) {
      const loadedId = dashboard.id;
      if (loadedId !== this._dashboardId) {
        this.props.findDashboard(this._projectId, this._dashboardId);
      }
    } else {
      this.props.clearDashboard();
    }
  }

  onSubmit = (value) => {
    const { history } = this.props;
    this.props.saveDashboard(this._projectId, value, this._isNew)
      .then(() => history.push(`/admin/projects/${this._projectId}/dashboards/${value.id}`));
  };

  render() {
    const { dashboard } = this.props;
    const title = this._isNew ? 'New Dashboard' : `Dashboard ${this._dashboardId}`;
    const isLoading = this._dashboardId && isEmpty(dashboard);
    const dashId = dashboard ? dashboard.generatedId : '';

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={title} />
        {isLoading && <span>Loading...</span>}
        {!isLoading
          && (
          <>
            <NewUiAlert
              type="adminDashboard"
              projectId={this._projectId}
              dashId={dashId}
            />
            <DashboardForm
              projectId={this._projectId}
              dashboard={dashboard}
              isNew={this._isNew}
              onSubmit={this.onSubmit}
            />
          </>
          )}
      </div>
    );
  }
}

DashboardEditPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  dashboard: PropTypes.object.isRequired,
  findDashboard: PropTypes.func.isRequired,
  saveDashboard: PropTypes.func.isRequired,
  clearDashboard: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({ dashboard: state.dashboard });

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findDashboard,
  saveDashboard,
  clearDashboard,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(DashboardEditPage);

export default wrapSubProjectEditPage(connectedPage);
