import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import FormButtons from '../../components/forms/FormButtons';
import FormInput from '../../components/forms/FormInput';
import FormElement from '../../components/forms/FormElement';
import ApiCallExample from '../../components/ApiCallExample';
import { inputTargetValue } from '../../utils/inputs';
import DashboardRowsEditor from '../../components/DashboardRowsEditor';
import { EditableTable } from '../../components/ParameterTables';

class DashboardForm extends PureComponent {
  static dashboardToState = (dashboard, isNew) => {
    if (dashboard.id) {
      const state = { ...dashboard };
      state.id = isNew ? '' : state.id;
      state.name = isNew ? '' : state.name;
      return state;
    }
    return { id: '', name: '' };
  };

  static mapPropsToState = (dashboard, isNew) => ({
    dashboard: DashboardForm.dashboardToState(dashboard, isNew),
    apiVisibility: true,
  });

  constructor(props) {
    super(props);

    this.state = DashboardForm.mapPropsToState(props.dashboard, props.isNew);
    this._nameInputTouched = !!props.dashboard.name;
    this._isNewDashboard = props.isNew;
  }

  // TODO: make work with state correctly for React 16
  componentDidUpdate(prevProps) {
    // We cannot use setState() because it performs a shallow merge of
    // previous state with the new one. But here we need entirely update state.
    if (this.props.projectId !== prevProps.projectId
      || this.props.dashboard !== prevProps.dashboard) {
      this.state = DashboardForm.mapPropsToState(this.props.dashboard, this.props.isNew);
    }
  }

  onInputChange = (event) => {
    const { target } = event;
    const value = inputTargetValue(target);

    const change = {};
    switch (target.name) {
      case 'id':
        change.id = value;
        if (!this._nameInputTouched) {
          change.name = value;
        }
        break;
      case 'name':
        this._nameInputTouched = true;
        change.name = value;
        break;
      default:
        change[target.name] = value;
    }

    this.changeDashboard(change);
  };

  onParameterUpdate = (paramIndex, param) => {
    const parameters = this.state.dashboard.parameters ? [...this.state.dashboard.parameters] : [];
    if (paramIndex < 0) {
      parameters.push(param);
    } else {
      parameters.splice(paramIndex, 1, param);
    }
    this.changeDashboard({ parameters });
  };

  onParameterDelete = (paramIndex) => {
    const parameters = this.state.dashboard.parameters ? [...this.state.dashboard.parameters] : [];
    parameters.splice(paramIndex, 1);
    this.changeDashboard({ parameters });
  };

  onRowsChange = (rows) => {
    this.changeDashboard({ rows });
  };

  onSubmit = (event) => {
    event.preventDefault();
    this.props.onSubmit(this.state.dashboard);
  };

  onJsonStateChange = (dashboard) => {
    this.setState({ dashboard });
  };

  onApiVisibilityChange = (apiVisibility) => {
    this.setState({ apiVisibility });
  };

  changeDashboard = (update) => {
    this.setState({ dashboard: { ...this.state.dashboard, ...update } });
  };

  render() {
    const { projectId } = this.props;

    const { dashboard, apiVisibility } = this.state;

    const {
      id, name, heightMultiplier, parameters, rows,
    } = dashboard;

    return (
      <div className="form-horizontal">
        <div className={apiVisibility ? 'col-lg-6 col-md-7' : 'col-lg-10 col-md-11'}>
          <FormInput
            type="text" name="id" label="Id" value={id} onChange={this.onInputChange}
            help="Unique dashboard identificator"
            disabled={!this._isNewDashboard}
          />
          <FormInput
            type="text" name="name" label="Name" value={name} onChange={this.onInputChange}
            help="This value will be displayed in Solomon UI"
          />
          <FormInput
            type="number"
            name="heightMultiplier" label="Height multiplier"
            value={heightMultiplier}
            onChange={this.onInputChange}
          />
          <FormElement label="Parameters">
            <EditableTable
              parameters={parameters || []}
              onDelete={this.onParameterDelete}
              onUpdate={this.onParameterUpdate}
            />
          </FormElement>
          <FormElement label="Rows">
            <DashboardRowsEditor rows={rows} onChange={this.onRowsChange} />
          </FormElement>
          <FormButtons onSubmit={this.onSubmit} />
        </div>

        <div className={apiVisibility ? 'col-lg-6 col-md-5' : 'col-lg-2 col-md-1'}>
          <ApiCallExample
            path={`/projects/${projectId}/dashboards`}
            value={dashboard}
            isNew={this._isNewDashboard}
            visible={apiVisibility}
            onChange={this.onJsonStateChange}
            onChangeVisibility={this.onApiVisibilityChange}
          />
        </div>
      </div>
    );
  }
}

DashboardForm.propTypes = {
  projectId: PropTypes.string.isRequired,
  dashboard: PropTypes.object.isRequired,
  isNew: PropTypes.bool.isRequired,
  onSubmit: PropTypes.func.isRequired,
};

export default DashboardForm;
