import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';
import isEmpty from 'lodash/isEmpty';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import LabeledValue from '../../components/LabeledValue';
import Panel from '../../components/Panel';
import ShowActions from '../../components/ShowActions';

import { deleteDashboard, findDashboard } from '../../store/reducers/dashboards/dashboard';
import { fetchDashboardParamSensors } from '../../store/reducers/paramSensors/dashboardParamSensorsPage';

import { ReadonlyTable } from '../../components/ParameterTables';
import DashboardRowsEditor from '../../components/DashboardRowsEditor';
import PreviewLink from '../../components/PreviewLink/PreviewLink';
import EntityInfo from '../../components/EntityInfo';
import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';
import { PROJECT_PERMISSIONS } from '../../auth/ProjectPermissions';
import MonitoringButton from '../old/components/ControlBar/MonitoringButton';
import { getMonitoringUiHost } from '../old/utils/graphToExpressionTransformations';
import { isPreOrProdIntEnv } from '../../utils/env';

class DashboardPage extends PureComponent {
  constructor(props) {
    super(props);
    const { projectId, dashboardId } = props.match.params;
    this._projectId = projectId;
    this._dashboardId = dashboardId;
  }

  componentDidMount() {
    this.props.findDashboard(this._projectId, this._dashboardId)
      .then((resp) => {
        if (resp && resp.payload) {
          this.props.fetchDashboardParamSensors(this._projectId, resp.payload.parameters);
        }
      });
  }

  componentDidUpdate(prevProps) {
    const { params } = this.props.match;
    const prevParams = prevProps.match.params;
    if (params.dashboardId !== prevParams.dashboardId) {
      this._projectId = params.projectId;
      this._dashboardId = params.dashboardId;
      this.props.findDashboard(this._projectId, this._dashboardId)
        .then((resp) => {
          if (resp && resp.payload) {
            this.props.fetchDashboardParamSensors(this._projectId, resp.payload.parameters);
          }
        });
    }
  }

  onDelete = (event) => {
    event.preventDefault();
    const { history } = this.props;
    this.props.deleteDashboard(this._projectId, this._dashboardId)
      .then((action) => {
        if (action) {
          history.push(`/admin/projects/${this._projectId}/dashboards`);
        }
      });
  };

  _dashboardContent = (dashboard, paramSensorsPage) => {
    if (isEmpty(dashboard)) {
      return <span>Loading...</span>;
    }

    const parametersTable = isEmpty(dashboard.parameters)
      ? '-'
      : <ReadonlyTable parameters={dashboard.parameters} />;

    const previewLink = (
      <PreviewLink
        type="dashboard"
        projectId={this._projectId}
        itemId={this._dashboardId}
        params={dashboard.parameters}
        sensorsPage={paramSensorsPage}
      />
    );

    return (
      <div>
        <Panel title="Generic options">
          <LabeledValue label="Name" value={dashboard.name} />
          <LabeledValue label="Height multiplier" value={dashboard.heightMultiplier} />
          <LabeledValue label="Parameters" value={parametersTable} />
          <EntityInfo entity={dashboard} />
          <LabeledValue label="" value={previewLink} />
        </Panel>
      </div>
    );
  };

  _dashboardRowsContent = (dashboard) => {
    if (isEmpty(dashboard.rows)) {
      return null;
    }

    return (
      <Panel title="Rows">
        <DashboardRowsEditor rows={dashboard.rows} readOnly />
      </Panel>
    );
  };

  render() {
    const { dashboard, paramSensorsPage, projectAuth } = this.props;

    if (isEmpty(dashboard)) {
      return <div>Loading...</div>;
    }

    const canEdit = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_UPDATE);
    const canDelete = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_DELETE);

    const newUiUrl = `https://${getMonitoringUiHost()}/projects/${
      this._projectId
    }/dashboards/${dashboard.generatedId}?utm_source=solomon_dashboard_open_from_admin`;

    return (
      <div>
        <Breadcrumb match={this.props.match} />

        <PageTitle title={`Dashboard ${this._dashboardId}`} />
        {isPreOrProdIntEnv() && (
          <div className="btn-toolbar table-toolbar">
            <div className="btn-group">
              <MonitoringButton href={newUiUrl} />
            </div>
          </div>
        )}

        <div className="row">
          <div className="col-xs-6">
            {this._dashboardContent(dashboard, paramSensorsPage)}
            <ShowActions
              editPath={`/admin/projects/${this._projectId}/dashboards/${this._dashboardId}/edit`}
              duplicatePath={`/admin/projects/${this._projectId}/dashboards/new?duplicateOf=${this._dashboardId}`}
              onDelete={this.onDelete}
              canEdit={canEdit}
              canChangeState={false}
              canDelete={canDelete}
            />
          </div>
          <div className="col-xs-6">
            {this._dashboardRowsContent(dashboard)}
          </div>
        </div>
      </div>
    );
  }
}

DashboardPage.propTypes = {
  match: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  dashboard: PropTypes.object.isRequired,
  paramSensorsPage: PropTypes.object.isRequired,
  projectAuth: PropTypes.object.isRequired,
  findDashboard: PropTypes.func.isRequired,
  deleteDashboard: PropTypes.func.isRequired,
  fetchDashboardParamSensors: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  dashboard: state.dashboard,
  paramSensorsPage: state.dashboardParamSensorsPage,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findDashboard,
  deleteDashboard,
  fetchDashboardParamSensors,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(DashboardPage);

export default wrapSubProjectReadPage(connectedPage);
