import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import isEmpty from 'lodash/isEmpty';
import isEqual from 'lodash/isEqual';
import JSONPretty from 'react-json-pretty';

import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';

import { formatSearch, parseSearch } from '../../utils/url';

import { clearData, loadData } from '../../store/reducers/data/sensorsData';

import WarningAlert from '../projects/alerts/WarningAlert';
import GraphWithLegend from './GraphWithLegend';
import TimeBar from './TimeBar/TimeBar';
import {
  computeRangeFromParams, mapAutoGraphConfigToDownsamplingParams,
  mapAutoGraphConfigToSearch,
  mapSearchToAutoGraphConfig,
} from './utils';
import SimpleTargetEditor from './SimpleTargetEditor/SimpleTargetEditor';
import ControlBar from './ControlBar';
import Selectors from '../../utils/Selectors';
import { AdminYasmAlert } from '../../components/YasmAlert';
import NewUiAlert from '../../components/NewUiAlert';
import { CURRENT_SEL_VERSION, parseSelVersionParam } from '../../utils/selVersion';
import { getMonitoringUiHost } from '../old/utils/graphToExpressionTransformations';
import { loadUserSettings } from '../../store/reducers/others/userSettings';

function isText(value) {
  if (!value) {
    return false;
  }
  try {
    return JSON.parse(value).queryViewMode === 'text';
  } catch (e) {
    console.error(e);
    return false;
  }
}

class AutoGraphPage extends PureComponent {
  static mapSearchToState(search, nowMillis) {
    const params = parseSearch(search);

    const b = params.get('b') || '1d';
    const e = params.get('e') || '';

    const { fromMillis, toMillis } = computeRangeFromParams(b, e, nowMillis);

    const expression = params.get('expression') || '';
    const selectors = params.get('selectors') || '';
    const forceCluster = params.get('forceCluster') || '';
    const versionParam = params.get('version') || '';
    const version = parseSelVersionParam(versionParam);

    let type = 'selectors';

    if (expression !== '') {
      type = 'expression';
    } else {
      type = 'selectors';
    }

    const useNewFormat = params.get('useNewFormat') || '';

    return {
      type,
      expression,
      selectors,
      forceCluster,
      config: mapSearchToAutoGraphConfig(params),
      b,
      e,
      fromMillis,
      toMillis,
      nowMillis,
      useNewFormat,
      version,
      versionParam,
    };
  }

  static mapStateToSearch(state) {
    const search = mapAutoGraphConfigToSearch(state.config);

    if (state.type === 'expression' && state.expression) {
      search.expression = state.expression;
    } else if (state.type === 'selectors' && state.selectors) {
      search.selectors = state.selectors;
    }

    if (state.b && state.b !== '1d') {
      search.b = state.b;
    }

    if (state.e) {
      search.e = state.e;
    }

    if (state.forceCluster) {
      search.forceCluster = state.forceCluster;
    }

    if (state.useNewFormat) {
      search.useNewFormat = state.useNewFormat;
    }

    if (state.versionParam && state.version !== CURRENT_SEL_VERSION) {
      search.version = state.versionParam;
    }

    return `?${formatSearch(search)}`;
  }

  static mapStateToParams(state) {
    const { fromMillis, toMillis } = state;

    const { type } = state;
    const value = state[type];

    let program;
    if (type === 'selectors') {
      try {
        program = Selectors.parse(value).format(true);
      } catch (e) {
        console.warn('failed to parse selectors', e);
      }
    } else {
      program = value;
    }

    const downsampling = mapAutoGraphConfigToDownsamplingParams(state.config);

    return {
      program,
      from: new Date(fromMillis).toISOString(),
      to: new Date(toMillis).toISOString(),
      downsampling,
      forceCluster: state.forceCluster,
      version: state.version,
    };
  }

  constructor(props) {
    super(props);
    this.state = AutoGraphPage.mapSearchToState(props.location.search, Date.now());
  }

  componentDidMount() {
    this._loadData();
  }

  componentDidUpdate(prevProps) {
    if (!isEqual(this.props.location.search, prevProps.location.search)) {
      this.setState(AutoGraphPage.mapSearchToState(this.props.location.search, Date.now()), () => {
        this._loadData();
      });
    }
  }

  componentWillUnmount() {
    this.props.clearData();
  }

  handleTargetsChange = ({ type, expression, selectors }) => {
    this.setState({ type, expression, selectors }, () => this._reloadData());
  };

  handleConfigChange = (config) => {
    this.setState({ config }, () => this._reloadData());
  };

  handleRangeChangeFromTimeBar = (b, e) => {
    this.setState({ b, e }, () => this._reloadData());
  };

  handleRangeChangeFromGraph = (fromMillis, toMillis) => {
    this.setState({
      b: new Date(fromMillis).toISOString(),
      e: new Date(toMillis).toISOString(),
    }, () => this._reloadData());
  };

  _reloadData = () => {
    this._refreshUrl();
  };

  _refreshUrl = () => {
    this.props.history.push(AutoGraphPage.mapStateToSearch(this.state));
  };

  _loadData = () => {
    const value = this.state[this.state.type];

    const useNewFormat = this.isUseNewFormat();

    if (value) {
      const params = AutoGraphPage.mapStateToParams(this.state);
      this.props.loadData(this.props.match.params.projectId, params, useNewFormat);
    } else {
      this.props.clearData();
    }
    this.props.loadUserSettings();
  };

  isUseNewFormat() {
    return this.state.useNewFormat === 'true';
  }

  render() {
    const { sensorsData } = this.props;

    let mainElement;

    if (sensorsData.loading) {
      mainElement = <div>Loading...</div>;
    } else if (sensorsData.error) {
      mainElement = (
        <WarningAlert title="Data error!" message={sensorsData.error} />
      );
    } else if (isEmpty(sensorsData.data)) {
      mainElement = null;
    } else {
      const result = sensorsData.data;

      if (!(result.timeseries
        || (result.vector && (result.vector.length === 0 || result.vector[0].timeseries)))) {
        mainElement = <JSONPretty json={result} />;
      } else {
        mainElement = (
          <GraphWithLegend
            results={[result]}
            config={this.state.config}
            fromMillis={this.state.fromMillis}
            toMillis={this.state.toMillis}
            onRangeChange={this.handleRangeChangeFromGraph}
          />
        );
      }
    }

    const targets = (
      <SimpleTargetEditor
        type={this.state.type}
        expression={this.state.expression}
        selectors={this.state.selectors}
        onChange={this.handleTargetsChange}
      />
    );

    let settings = null;

    if (this.state.expression || this.state.selectors) {
      settings = (
        <div className="row" style={{ marginTop: '10px' }}>
          <div className="col-sm-12">
            <div className="pull-left">
              <ControlBar
                config={this.state.config}
                fromMillis={this.state.fromMillis}
                toMillis={this.state.toMillis}
                onChange={this.handleConfigChange}
              />
            </div>
            <div className="pull-right">
              <TimeBar
                b={this.state.b}
                e={this.state.e}
                fromMillis={this.state.fromMillis}
                toMillis={this.state.toMillis}
                onRangeChange={this.handleRangeChangeFromTimeBar}
              />
            </div>
          </div>
        </div>
      );
    }

    const { projectId } = this.props.match.params;

    const type = isText(this.props.uiSettings) ? 'text' : 's';
    const explorerUrl = `https://${getMonitoringUiHost()}/projects/${projectId}/explorer/queries?q.0.${type}=${encodeURIComponent(
      this.state.expression || `{${this.state.selectors}}` || '',
    )}`;

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title="Auto graph" />
        <AdminYasmAlert projectId={projectId} />
        <NewUiAlert type="adminAutoGraph" projectId={projectId} url={explorerUrl} />
        {targets}
        {settings}
        {mainElement}
      </div>
    );
  }
}

AutoGraphPage.propTypes = {
  history: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  match: PropTypes.object.isRequired,
  sensorsData: PropTypes.object.isRequired,
  uiSettings: PropTypes.string,
  loadData: PropTypes.func.isRequired,
  clearData: PropTypes.func.isRequired,
  loadUserSettings: PropTypes.func.isRequired,
};

AutoGraphPage.defaultProps = {
  uiSettings: '',
};

const mapStateToProps = (state) => ({
  sensorsData: state.sensorsData,
  uiSettings: state.userSettings.data.settings.ui,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadData,
  clearData,
  loadUserSettings,
}, dispatch);

export default wrapSubProjectReadPage(
  connect(mapStateToProps, mapDispatchToProps)(AutoGraphPage),
);
