import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import Dropdown from 'react-bootstrap/es/Dropdown';
import MenuItem from 'react-bootstrap/es/MenuItem';

import GraphConfigureModal from './GraphConfigureModal';
import QuickCopyButton from '../../components/QuickCopyButton/QuickCopyButton';
import UrlUtils from '../../utils/url/UrlUtils';
import BrowserUtils from '../../utils/BrowserUtils';
import UserLinksBasic from '../../utils/UserLinksBasic';
import { formatISO8601 } from '../../utils/iso8601';
import { knownDcs } from '../../utils/env';

function createStaticUrl(fromMillis, toMillis) {
  const baseHashUrl = UrlUtils.removeQueryArgs(BrowserUtils.locationHref(),
    UserLinksBasic.B_QA, UserLinksBasic.E_QA);

  return UrlUtils.addTwoParameters(
    baseHashUrl,
    UserLinksBasic.B_QA, formatISO8601(fromMillis),
    UserLinksBasic.E_QA, formatISO8601(toMillis),
  );
}

class ControlBar extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { showConfigureModal: false };
  }

  handleConfigureModalShow = (event) => {
    event.preventDefault();
    this.setState({ showConfigureModal: true });
  };

  handleConfigureModalApply = (config) => {
    this.handleConfigureModalHide();
    this.props.onChange(config);
  };

  handleConfigureModalHide = () => {
    this.setState({ showConfigureModal: false });
  };

  render() {
    const staticUrl = createStaticUrl(this.props.fromMillis, this.props.toMillis);

    const forceCluster = BrowserUtils.getLogicalQueryArgOrEmpty('forceCluster');

    const forceClusterOptions = knownDcs.map((dc) => dc.id).map((dc) => {
      if (dc === forceCluster) {
        return null;
      }

      const href = UrlUtils.updateParameter(
        BrowserUtils.getLogicalUrlAfterHost(),
        UserLinksBasic.FORCE_CLUSTER_PARAM,
        dc,
      );

      return (
        <MenuItem key={`force_to_${dc}`} href={href}>
          forceCluster=
          {dc}
        </MenuItem>
      );
    });

    if (forceCluster) {
      const href = UrlUtils.removeQueryArgs(
        BrowserUtils.getLogicalUrlAfterHost(),
        UserLinksBasic.FORCE_CLUSTER_PARAM,
      );
      forceClusterOptions.push((
        <MenuItem key="do_not_force" href={href}>
          Do not force cluster
        </MenuItem>
      ));
    }

    let showMoreDropdown = null;

    if (forceClusterOptions.length > 0) {
      showMoreDropdown = (
        <div className="btn-group">
          <Dropdown id="admin-more-btn">
            <Dropdown.Toggle style={{ boxShadow: 'none', background: 'transparent', border: 'transparent' }} noCaret>
              <i className="glyphicon glyphicon-option-vertical" />
            </Dropdown.Toggle>
            <Dropdown.Menu>
              {forceClusterOptions}
            </Dropdown.Menu>
          </Dropdown>
        </div>
      );
    }

    return (
      <div className="btn-toolbar">
        <div className="btn-group">
          <button
            type="button"
            className="btn btn-default"
            onClick={this.handleConfigureModalShow}
          >
            Configure
          </button>
        </div>
        {showMoreDropdown}
        <div className="btn-group">
          <QuickCopyButton url={staticUrl} />
        </div>
        <GraphConfigureModal
          config={this.props.config}
          onChange={this.handleConfigureModalApply}
          onCancel={this.handleConfigureModalHide}
          isOpen={this.state.showConfigureModal}
        />
      </div>
    );
  }
}

ControlBar.propTypes = {
  config: PropTypes.object.isRequired,
  fromMillis: PropTypes.number.isRequired,
  toMillis: PropTypes.number.isRequired,
  onChange: PropTypes.func.isRequired,
};

export default ControlBar;
