/* eslint-disable jsx-a11y/label-has-for */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import isEqual from 'lodash/isEqual';

import Modal from '../../components/Modal';
import FormInput from '../../components/forms/FormInput';
import { inputTargetValue } from '../../utils/inputs';
import FormCheckbox from '../../components/forms/FormCheckbox';
import { TabPane, Tabs } from '../../components/Tabs';
import FormRadioGroup from '../graphs/FormRadioGroup';
import NumberFormatSelector from '../graphs/NumberFormatSelector';
import FormElement from '../../components/forms/FormElement';

const INTERPOLATE_OPTIONS = [
  { value: 'linear', title: 'Linear' },
  { value: 'left', title: 'Left' },
  { value: 'right', title: 'Right' },
  { value: 'none', title: 'None' },
];

const COLOR_SCHEME_OPTIONS = [
  { value: 'auto', title: 'Auto' },
  { value: 'default', title: 'Default' },
  { value: 'gradient', title: 'Gradient' },
];

const SCALE_OPTIONS = [
  { value: 'natural', title: 'Natural' },
  { value: 'log', title: 'Logarithmic' },
];

const DOWNSAMPLING_MODE_OPTIONS = [
  { value: 'auto', title: 'Auto' },
  { value: 'byPoints', title: 'By points' },
  { value: 'byGrid', title: 'By grid' },
  { value: 'off', title: 'Off' },
];

const DOWNSAMPLING_AGGR_OPTIONS = [
  { value: 'default', title: 'Default' },
  { value: 'max', title: 'Max' },
  { value: 'min', title: 'Min' },
  { value: 'sum', title: 'Sum' },
  { value: 'avg', title: 'Avg' },
  { value: 'last', title: 'Last' },
  { value: 'count', title: 'Count' },
];

const DOWNSAMPLING_FILL_OPTIONS = [
  { value: 'null', title: 'Null' },
  { value: 'none', title: 'None' },
  { value: 'previous', title: 'Previous' },
];

class GraphConfigureModal extends PureComponent {
  constructor(props) {
    super(props);

    this.state = props.config || {};
  }

  // TODO: make work with state correctly for React 16
  componentDidUpdate(prevProps) {
    if (!isEqual(this.props.config, prevProps.config)) {
      this.state = this.props.config || {};
    }
  }

  handleInput = (event) => {
    const { name } = event.target;
    const value = inputTargetValue(event.target);
    this.setState({ [name]: value });
  };

  handleNumberFormatChange = (numberFormat) => {
    this.setState({ numberFormat });
  };

  handleSelect = (event) => {
    this.setState({ [event.target.name]: event.target.value });
  };

  handleApplyClick = () => {
    this.props.onChange(this.state);
  };

  handleCloseClick = () => {
    this.props.onCancel();
  };

  render() {
    const { downsamplingMode } = this.state;

    return (
      <Modal
        title="Configure graph"
        okTitle="Apply"
        cancelTitle="Close"
        onOk={this.handleApplyClick}
        onCancel={this.handleCloseClick}
        isOpen={this.props.isOpen}
      >
        <div className="graph-configure-form">
          <Tabs>
            <TabPane label="Display">
              <div className="form-horizontal">
                <FormCheckbox
                  name="stack"
                  label="Stack"
                  value={this.state.stack}
                  onChange={this.handleInput}
                />
                <FormCheckbox
                  name="norm"
                  label="Normalize"
                  value={this.state.norm}
                  onChange={this.handleInput}
                />
                <FormRadioGroup
                  name="interpolate"
                  label="Interpolate"
                  value={this.state.interpolate}
                  defaultValue="linear"
                  options={INTERPOLATE_OPTIONS}
                  onChange={this.handleSelect}
                />
                <FormRadioGroup
                  name="cs"
                  label="Color scheme"
                  options={COLOR_SCHEME_OPTIONS}
                  value={this.state.cs}
                  defaultValue="auto"
                  onChange={this.handleSelect}
                />
                {this.state.cs === 'gradient' && (
                  <FormInput
                    type="text"
                    name="green"
                    label="Green"
                    value={this.state.green}
                    onChange={this.handleInput}
                  />
                )}
                {this.state.cs === 'gradient' && (
                  <FormInput
                    type="text"
                    name="yellow"
                    label="Yellow"
                    value={this.state.yellow}
                    onChange={this.handleInput}
                  />
                )}
                {this.state.cs === 'gradient' && (
                  <FormInput
                    type="text"
                    name="red"
                    label="Red"
                    value={this.state.red}
                    onChange={this.handleInput}
                  />
                )}
                {this.state.cs === 'gradient' && (
                  <FormInput
                    type="text"
                    name="violet"
                    label="Violet"
                    value={this.state.violet}
                    onChange={this.handleInput}
                  />
                )}
                <FormCheckbox
                  name="dropNaN"
                  label="Drop NaN values"
                  value={this.state.dropNaN}
                  onChange={this.handleInput}
                />
              </div>
            </TabPane>
            <TabPane label="Axis">
              <div className="form-horizontal">
                <FormRadioGroup
                  label="Scale"
                  name="scale"
                  value={this.state.scale}
                  defaultValue="natural"
                  options={SCALE_OPTIONS}
                  onChange={this.handleSelect}
                />
                <FormInput
                  type="text"
                  label="Min"
                  name="min"
                  value={this.state.min}
                  onChange={this.handleInput}
                />
                <FormInput
                  type="text"
                  label="Max"
                  name="max"
                  value={this.state.max}
                  onChange={this.handleInput}
                />
                <FormElement label="Number format">
                  <NumberFormatSelector
                    value={this.state.numberFormat}
                    onChange={this.handleNumberFormatChange}
                  />
                </FormElement>
              </div>
            </TabPane>
            <TabPane label="Downsampling">
              <div className="form-horizontal">
                <FormRadioGroup
                  label="Downsampling"
                  name="downsamplingMode"
                  value={downsamplingMode}
                  defaultValue="auto"
                  options={DOWNSAMPLING_MODE_OPTIONS}
                  onChange={this.handleSelect}
                />
                {this.state.downsamplingMode === 'byPoints' && (
                  <FormInput
                    type="number"
                    label="Max points"
                    name="maxPoints"
                    value={this.state.maxPoints}
                    onChange={this.handleInput}
                  />
                )}
                {this.state.downsamplingMode === 'byGrid' && (
                  <FormInput
                    type="text"
                    label="Grid"
                    name="grid"
                    value={this.state.grid}
                    onChange={this.handleInput}
                  />
                )}
                {this.state.downsamplingMode !== 'off' && (
                  <FormRadioGroup
                    label="Aggregation"
                    name="downsamplingAggr"
                    value={this.state.downsamplingAggr}
                    defaultValue="default"
                    options={DOWNSAMPLING_AGGR_OPTIONS}
                    onChange={this.handleSelect}
                  />
                )}
                {this.state.downsamplingMode !== 'off' && (
                  <FormRadioGroup
                    label="Fill option"
                    name="downsamplingFill"
                    value={this.state.downsamplingFill}
                    defaultValue="null"
                    options={DOWNSAMPLING_FILL_OPTIONS}
                    onChange={this.handleSelect}
                  />
                )}
              </div>
            </TabPane>
          </Tabs>
        </div>
      </Modal>
    );
  }
}

GraphConfigureModal.propTypes = {
  config: PropTypes.object.isRequired,
  onChange: PropTypes.func.isRequired,
  onCancel: PropTypes.func.isRequired,
  isOpen: PropTypes.bool.isRequired,
};

export default GraphConfigureModal;
