import YaxisCommon from '../../solomon-graph/common/YaxisCommon';
import Multiplier from '../../utils/Multiplier';
import { computeMinAbs } from '../../solomon-graph/common/AggrFunctions';
import { lineDataToAggrValues } from '../old/OldGraphPage/GraphModeEngine';
import ChartLines from '../../solomon-graph/lines/ChartLines';
import ChartBarsCommon from '../../solomon-graph/bars/ChartBarsCommon';
import GraphConfUtils from '../old/OldGraphPage/utils/GraphConfUtils';
import UserLinksBasic from '../../utils/UserLinksBasic';
import NumberFormatter from '../../utils/NumberFormatter';
import { parseNumberFormat } from '../graphs/NumberFormatUtils';

class GraphFormatter {
  static makeFromGraphData(conf, lines, chartHeight) {
    try {
      return GraphFormatter.makeFromGraphDataImpl(conf, lines, chartHeight);
    } catch (e) {
      console.error('Failed to make formatter', e);
      return new NumberFormatter(3, new Multiplier(0, '', 0));
    }
  }

  static makeFromGraphDataImpl(conf, lines, chartHeight) {
    const scaleData = GraphFormatter.computeScaleData(conf, lines);
    // eslint-disable-next-line max-len
    const minYaxisTickAbs = GraphFormatter.computeMinYaxisTickAbsOrZero(conf, scaleData, chartHeight);

    let yaxisPoint;
    if (minYaxisTickAbs === 0) {
      // no ticks in range or zero tick only:
      // ok to take range max and halve it to avoid bad boundaries
      yaxisPoint = Math.max(Math.abs(scaleData.getMin()), Math.abs(scaleData.getMax())) / 2;

      if (yaxisPoint === 0) {
        yaxisPoint = 1;
      }
    } else {
      yaxisPoint = minYaxisTickAbs;
    }

    return GraphFormatter.makeFormatterForYaxisTick(conf, yaxisPoint);
  }

  static makeFormatterForYaxisTick(conf, yaxisTick) {
    const parsed = parseNumberFormat(conf[UserLinksBasic.GRAPH_NUMBER_FORMAT] || '');

    const multiplierForNumber = Multiplier.getMultiplierForNumber(yaxisTick);

    const multiplier = Multiplier.forSuffix(parsed.multiplier, multiplierForNumber);

    return new NumberFormatter(parsed.precision, multiplier);
  }

  static computeMinYaxisTickAbsOrZero(conf, scaleData, chartHeight) {
    const mainGraphMode = conf.graphMode || 'graph';

    if (mainGraphMode === 'graph' || mainGraphMode === 'bars' || mainGraphMode === 'distribution') {
      const isLog = conf.scale === 'log';

      const yaxisTicks = YaxisCommon.makeYAxisTicks(
        scaleData,
        YaxisCommon.MIN_YAXIS_TICK_HEIGHT,
        chartHeight,
        isLog,
      );
      return computeMinAbs(yaxisTicks, (tick) => tick.y);
    }

    return 0;
  }

  static computeScaleData(conf, lines) {
    const chartConf = GraphConfUtils.toChartConf(conf);

    if (!conf.graphMode || conf.graphMode === UserLinksBasic.GRAPH_MODE_GRAPH) {
      return GraphFormatter.computeScaleDataFromLines(chartConf, lines);
    }

    const aggrValues = lineDataToAggrValues(lines);
    return ChartBarsCommon.computeAggrYAxisMinMax(aggrValues, chartConf);
  }

  static computeScaleDataFromLines(chartConf, lines) {
    const leftLines = lines.filter((lineData) => !lineData.yaxisConf || lineData.yaxisConf.positionValue === 'left');
    return ChartLines.computeLinesYAxisMinMaxPlusSome(leftLines, chartConf);
  }
}

export default GraphFormatter;
