import React, { Component } from 'react';
import PropTypes from 'prop-types';
import isEqual from 'lodash/isEqual';

import { fixSeries } from '../../utils/data';
import LineChart from './LineChart';
import TableLegend from './TableLegend/TableLegend';
import PieChart from './PieChart';
import GraphFormatter from './GraphFormatter';
import { computeMinAbs } from '../../solomon-graph/common/AggrFunctions';

function getValuesFromData(data) {
  return data.map((point) => point[1])
    .filter((value) => !isNaN(value) && Number.isFinite(value));
}

function getValuesFromLine(line) {
  return line.values.filter((value) => !isNaN(value) && Number.isFinite(value));
}

function getLastPointValue(data) {
  const values = getValuesFromData(data);
  return values.length === 0 ? NaN : values[values.length - 1];
}

function convertLinesToAggrs(highchartsLines, func) {
  return highchartsLines.map((line) => ({
    ...line,
    data: null,
    y: func(line.data),
  }))
    .filter((s) => !isNaN(s.y) && Number.isFinite(s.y) && s.y !== 0)
    .map((s, i) => ({ ...s, index: i }));
}

function computeCommonChartProps(results, config) {
  const series = results.map((result) => {
    if (result.timeseries) {
      return [result.timeseries];
    }

    if (result.vector) {
      return result.vector.map((value) => value.timeseries).filter(Boolean);
    }

    return [];
  })
    .reduce((a, b) => a.concat(b));

  let lineSeries;
  let aggrSeries;
  let tableRows;
  let tableColumns;

  if (series) {
    let colorSchemeParams = null;

    if (config) {
      colorSchemeParams = {
        type: config.cs,
        green: config.green,
        yellow: config.yellow,
        red: config.red,
        violet: config.violet,
      };
    }

    const result = fixSeries(series, { colorScheme: colorSchemeParams });
    lineSeries = result.highchartsLines;
    tableColumns = result.columns;

    aggrSeries = convertLinesToAggrs(lineSeries, getLastPointValue);

    tableRows = result.lines.map((line, index) => {
      const pointValues = getValuesFromLine(line);

      const max = pointValues.length === 0 ? NaN : pointValues.reduce((a, b) => Math.max(a, b));
      const min = pointValues.length === 0 ? NaN : pointValues.reduce((a, b) => Math.min(a, b));
      const sum = pointValues.length === 0 ? NaN : pointValues.reduce((a, b) => a + b);
      const avg = pointValues.length === 0 ? NaN : sum / pointValues.length;
      const last = pointValues.length === 0 ? NaN : pointValues[pointValues.length - 1];

      const values = [min, max, last, avg, sum];

      return {
        id: index,
        names: line.legendNames,
        color: line.color,
        values,
        formattedValues: [],
      };
    });

    for (let i = 0; i < 5; ++i) {
      const minValue = computeMinAbs(tableRows, (row) => row.values[i]);

      const formatter = GraphFormatter.makeFormatterForYaxisTick(config, minValue);

      tableRows.forEach((row) => {
        const value = row.values[i];
        const formattedValue = formatter.format(value);
        row.formattedValues.push(formattedValue);
      });
    }
  } else {
    lineSeries = [];
    aggrSeries = [];
    tableRows = [];
    tableColumns = [];
  }

  return {
    lineSeries, aggrSeries, tableRows, tableColumns,
  };
}

class GraphWithLegend extends Component {
  shouldComponentUpdate(nextProps) {
    return this.props.results !== nextProps.results
      || !isEqual(this.props.config, nextProps.config);
  }

  render() {
    const {
      results, config, fromMillis, toMillis,
    } = this.props;

    const {
      lineSeries, aggrSeries, tableRows, tableColumns,
    } = computeCommonChartProps(results, config);

    const tableColumnNames = tableColumns.map((c) => c.title);

    return (
      <div>
        <LineChart
          series={lineSeries}
          config={config}
          fromMillis={fromMillis}
          toMillis={toMillis}
          onChangeRange={this.props.onRangeChange}
        />
        <div className="legend-and-secondary-graph-row">
          <div className="secondary-graph-placeholder">
            <PieChart series={aggrSeries} />
          </div>
          <div className="legend">
            <TableLegend
              nameTitles={tableColumnNames}
              valueTitles={['Min', 'Max', 'Last', 'Avg', 'Sum']}
              rows={tableRows}
            />
          </div>
        </div>
      </div>
    );
  }
}

GraphWithLegend.propTypes = {
  results: PropTypes.array.isRequired,
  config: PropTypes.object.isRequired,
  fromMillis: PropTypes.number.isRequired,
  toMillis: PropTypes.number.isRequired,
  onRangeChange: PropTypes.func.isRequired,
};

export default GraphWithLegend;
