import React, { Component } from 'react';
import PropTypes from 'prop-types';
import isEqual from 'lodash/isEqual';

import HighchartsChart from '../../components/HighchartsChart/HighchartsChart';
import parseKmg from '../../utils/Kmg';

import './style.css';

function convertParamsToHighchartsOptions(series, config, fromMillis, toMillis, onRangeChange) {
  const highchartsOptions = {
    title: '',
    series,
    legend: { enabled: false },
    xAxis: {
      type: 'datetime',
      gridLineWidth: 0.5,
      gridLineColor: '#a9a9a9',
      gridZIndex: 4,
      min: fromMillis,
      max: toMillis,
    },
    yAxis: {
      title: '',
      gridLineWidth: 0.5,
      gridLineColor: '#a9a9a9',
      endOnTick: false,
      gridZIndex: 4,
      reversedStacks: false,
    },
    chart: {
      height: '400px',
      type: 'line',
      zoomType: 'x',
      events: {
        selection(event) {
          const { min, max } = event.xAxis[0];

          if (min !== undefined && max !== undefined) {
            const roundMin = Math.round(min);
            const roundMax = Math.round(max);
            onRangeChange(roundMin, roundMax);
          }

          return false;
        },
      },
    },
    plotOptions: {
      line: {},
      area: {
        fillOpacity: 1,
      },
      column: {},
      scatter: {},
      series: {
        animation: false,
        marker: {
          radius: 0,
        },
        lineWidth: 1,
      },
    },
    tooltip: {
      shared: true,
      animation: false,
    },
    credits: {
      enabled: false,
    },
    time: {
      useUTC: false,
    },
  };

  let type = null;

  if (config.interpolate === 'none') {
    if (config.stack) {
      type = 'column';
    } else {
      type = 'scatter';
    }
  } else if (config.stack) {
    type = 'area';
  } else {
    type = 'line';
  }

  highchartsOptions.chart.type = type;

  const plotOptions = highchartsOptions.plotOptions[type];

  if (config.norm) {
    plotOptions.stacking = 'percent';
  } else if (config.stack) {
    plotOptions.stacking = 'normal';
  }

  if (config.interpolate === 'left' || config.interpolate === 'right') {
    plotOptions.step = config.interpolate;
  }

  if (type === 'scatter') {
    highchartsOptions.plotOptions.series.marker.radius = 5;
  }

  if (config.max) {
    const parsedMax = parseKmg(config.max);

    if (!isNaN(parsedMax)) {
      highchartsOptions.yAxis.max = parsedMax;
    }
  }

  if (config.min) {
    const parsedMin = parseKmg(config.min);
    if (!isNaN(parsedMin)) {
      highchartsOptions.yAxis.min = parsedMin;
    }
  }

  if (config.scale === 'log') {
    highchartsOptions.yAxis.type = 'logarithmic';
  }

  highchartsOptions.plotOptions.series.connectNulls = !!config.dropNaN !== !!config.stack;

  return highchartsOptions;
}

class LineChart extends Component {
  shouldComponentUpdate(nextProps) {
    return this.props.series !== nextProps.series
      || isEqual(this.props.config, nextProps.config);
  }

  render() {
    const {
      series, config, fromMillis, toMillis, onChangeRange,
    } = this.props;

    if (!series || series.length === 0) {
      return null;
    }

    const highchartsOptions = convertParamsToHighchartsOptions(
      series, config, fromMillis, toMillis, onChangeRange,
    );

    return <HighchartsChart options={highchartsOptions} />;
  }
}

LineChart.propTypes = {
  series: PropTypes.array,
  config: PropTypes.object.isRequired,
  fromMillis: PropTypes.number.isRequired,
  toMillis: PropTypes.number.isRequired,
  onChangeRange: PropTypes.func.isRequired,
};

LineChart.defaultProps = {
  series: [],
};

export default LineChart;
