import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import capitalize from 'lodash/capitalize';

import RadioButtons from '../../../components/RadioButtons';

import './style.css';

class SimpleTargetEditor extends PureComponent {
  static mapPropsToState(props) {
    return {
      type: props.type,
      expression: props.expression,
      selectors: props.selectors,
    };
  }

  static mapStateToChange(state) {
    return {
      type: state.type,
      expression: state.expression || '',
      selectors: state.selectors || '',
    };
  }

  constructor(props) {
    super(props);

    this.state = { ...SimpleTargetEditor.mapPropsToState(props) };
  }

  componentDidUpdate(prevProps) {
    if (this.props.type !== prevProps.type
      || this.props.expression !== prevProps.expression
      || this.props.selectors !== prevProps.selectors) {
      this.setState(SimpleTargetEditor.mapPropsToState(this.props));
    }
  }

  handleTypeChange = (type) => {
    this.setState({ type });
  };

  handleTextAreaValueKeyDown = (event) => {
    if (event.which === 13 && event.ctrlKey) {
      event.preventDefault();

      this.props.onChange(SimpleTargetEditor.mapStateToChange(this.state));
    }
  };

  handleValueChange = (event) => {
    this.setState({ [this.state.type]: event.target.value });
  };

  handleValueKeyPress = (event) => {
    if (event.which === 13) {
      event.preventDefault();

      this.props.onChange(SimpleTargetEditor.mapStateToChange(this.state));
    }
  };

  handleRunClick = (event) => {
    event.preventDefault();
    this.props.onChange(SimpleTargetEditor.mapPropsToState(this.state));
  };

  render() {
    const { type } = this.state;

    const value = this.state[type];

    let mainBlock = null;

    if (type === 'expression') {
      mainBlock = (
        <div style={{ display: 'flex', flexDirection: 'row' }}>
          <textarea
            className="form-control"
            value={value}
            placeholder={`${capitalize(type)}...`}
            onChange={this.handleValueChange}
            onKeyDown={this.handleTextAreaValueKeyDown}
            style={{ fontFamily: 'monospace', resize: 'vertical' }}
          />
          <div style={{ marginLeft: '5px' }}>
            <button type="button" className="btn btn-default" onClick={this.handleRunClick}>
              Run
            </button>
          </div>
        </div>
      );
    } else {
      mainBlock = (
        <input
          className="form-control"
          value={value}
          placeholder={`${capitalize(type)}...`}
          onChange={this.handleValueChange}
          onKeyPress={this.handleValueKeyPress}
        />
      );
    }

    return (
      <div className="simple-target-editor">
        <div className="simple-target-editor__type-selector">
          <RadioButtons
            choices={['selectors', 'expression']}
            labels={['Selectors', 'Expression']}
            selected={type}
            onSelect={this.handleTypeChange}
          />
        </div>
        <div className="simple-target-editor__target">
          {mainBlock}
        </div>
      </div>
    );
  }
}

SimpleTargetEditor.propTypes = {
  type: PropTypes.oneOf(['selectors', 'expression']).isRequired,
  expression: PropTypes.string.isRequired,
  selectors: PropTypes.string.isRequired,
  onChange: PropTypes.func.isRequired,
};

export default SimpleTargetEditor;
